/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.http.api.domain.entity;

import org.mule.runtime.api.util.MultiMap;

import java.io.IOException;
import java.nio.ByteBuffer;

/**
 * An {@link HttpEntity} that can be fed after it was created. Implementers can leverage it for non-blocking streaming and for
 * trailers sending.
 *
 * @since 4.11
 */
public interface FeedableHttpEntity extends HttpEntity {

  /**
   * Writes a fragment of the body to this entity abstraction.
   *
   * @param data buffer containing the data.
   * @throws IOException if an error occurs while writing the data.
   */
  void feed(ByteBuffer data) throws IOException;

  /**
   * Signals the entity with an error
   *
   * @param exception the error.
   */
  void error(Exception exception);

  /**
   * After {@link #feed}ing all the data fragments, user must complete the entity by calling this method of
   * {@link #completeWithTrailers}.
   *
   * @throws IOException if an error occurs during the completion of the entity.
   */
  void complete() throws IOException;

  /**
   * After {@link #feed}ing all the data fragments, user must complete the entity by calling this method of {@link #complete}.
   *
   * @param trailers the trailers.
   *
   * @throws IOException if an error occurs during the completion of the entity.
   */
  void completeWithTrailers(MultiMap<String, String> trailers) throws IOException;
}
