/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api.el;


import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.message.api.MuleEventMetadataType;
import org.mule.metadata.message.api.MuleEventMetadataTypeBuilder;

import java.io.InputStream;
import java.util.List;
import java.util.Map;

/**
 * Resolves the metadata for a specific expression language.
 */
public interface ExpressionLanguageMetadataTypeResolver {

  /**
   * Infers the expected input mule event type {@link MuleEventMetadataType} for specified output type with the given script.
   *
   * @param expression The scripting text.
   * @param output     The expected output type
   * @param builder    The builder to be used to build the event type
   * @param callback   The callback
   */
  void getInputType(String expression, MetadataType output, MuleEventMetadataTypeBuilder builder, MessageCallback callback);

  /**
   * Returns the result type expression when invoked with the given {@link TypeBindings}.
   *
   * @param typeBindings The script bindings
   * @param expression   The scripting text.
   * @param callback     The callback
   * @return The return type of the expression.
   */
  MetadataType getOutputType(TypeBindings typeBindings, String expression, MessageCallback callback);


  /**
   * Returns the result type expression when invoked with the given {@link TypeBindings}.
   *
   * @param typeBindings   The script bindings
   * @param expression     The scripting text.
   * @param outputMimeType The output mimeType of the expression
   * @param callback       The callback
   * @return The return type of the expression.
   */
  MetadataType getOutputType(TypeBindings typeBindings, String expression, String outputMimeType, MessageCallback callback);

  /**
   * Infers the metadata out of a sample data
   *
   * @param sample           The sample data to be use
   * @param readerProperties The configuration properties to read the sample data
   * @param mimeType         The mimeType of the sample data
   * @return The infered MetadataType
   */
  MetadataType getMetadataFromSample(InputStream sample, Map<String, Object> readerProperties, String mimeType);

  /**
   * Returns if the assignment type can be assigned to the expected type
   *
   * @param assignment The type to be assigned
   * @param expected   The expected type
   * @param callback   Callback for error messages. All the reasons of why it was not able to assign
   * @return True if it can be assign
   */
  boolean isAssignable(MetadataType assignment, MetadataType expected, MessageCallback callback);

  /**
   * Returns the substitution that needs to be done in order for this two types can be assigned
   *
   * @param assignment The assignment type
   * @param expected   The expected type
   * @param callback   The callback for errors and warnings
   * @return The substitution
   */
  Map<String, MetadataType> resolveAssignment(MetadataType assignment, MetadataType expected, MessageCallback callback);

  /**
   * Returns a new type with the substitution being applied
   *
   * @param assignment   The type to be substituted
   * @param substitution The substitution
   * @return The new type
   */
  MetadataType substitute(MetadataType assignment, Map<String, MetadataType> substitution);

  /**
   * Unify all the specified types into one type. It will remove duplications and return a UnionType only if required
   *
   * @param metadataTypes The types to be unified
   * @return The new type
   */
  MetadataType unify(List<MetadataType> metadataTypes);

  /**
   * Returns the default instance for this Type Resolver
   *
   * @return The default instance
   */
  static ExpressionLanguageMetadataTypeResolver getInstance() {
    return ExpressionLanguageMetadataTypeResolverProvider.getInstance();
  }

  static ExpressionLanguageMetadataTypeResolver getInstance(ClassLoader loader) {
    return ExpressionLanguageMetadataTypeResolverProvider.getInstance(loader);
  }

  /**
   * Callback from the resolver
   */
  interface MessageCallback {

    /**
     * Is called when a warning message happens while resolving metadata
     *
     * @param message  The message
     * @param location The location of the message
     */
    void warning(String message, MessageLocation location);

    /**
     * Is called when a error message happens while resolving metadata
     *
     * @param message  The message
     * @param location The location of the message
     */
    void error(String message, MessageLocation location);
  }


  /**
   * Represents a message location
   */
  class MessageLocation {

    private MessagePosition startPosition;
    private MessagePosition endPosition;

    public MessageLocation(MessagePosition startPosition, MessagePosition endPosition) {
      this.startPosition = startPosition;
      this.endPosition = endPosition;
    }

    /**
     * The start position of this location
     *
     * @return the position
     */
    public MessagePosition getStartPosition() {
      return this.startPosition;
    }

    /**
     * The end position of this location
     *
     * @return the position
     */
    public MessagePosition getEndPosition() {
      return this.endPosition;
    }
  }

  /**
   * A position with the line , column and offset in a document
   */
  class MessagePosition {

    private int line;
    private int column;
    private int offset;

    public MessagePosition(int line, int column, int offset) {
      this.line = line;
      this.column = column;
      this.offset = offset;
    }

    public int getLine() {
      return this.line;
    }

    public int getColumn() {
      return this.column;
    }

    public int getOffset() {
      return this.offset;
    }
  }

}
