/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectType;

import java.util.Optional;
import java.util.Set;


/**
 * Models a Message Type with its payload and attributes.
 *
 * @see MuleEventMetadataType
 * @since 1.0
 */
public interface MessageMetadataType extends ObjectType {

  static MessageMetadataTypeBuilder builder() {
    return new MessageMetadataTypeBuilder();
  }

  /**
   * The payload metadata types
   *
   * @return The payload type
   */
  Optional<MetadataType> getPayloadType();

  /**
   * The payload annotations
   *
   * @return The set of payload annotations
   */
  Optional<Set<TypeAnnotation>> getPayloadAnnotations();

  /**
   * The payload annotations of provided annotation class
   *
   * @param annotation class
   * @return optional annotation
   */
  <T extends TypeAnnotation> Optional<T> getPayloadAnnotation(Class<T> annotation);

  /**
   * The attributes metadata type
   *
   * @return The attribute type
   */
  Optional<MetadataType> getAttributesType();

  /**
   * The attributes annotations
   *
   * @return The set of attributes annotations
   */
  Optional<Set<TypeAnnotation>> getAttributesAnnotations();

  /**
   * The attributes annotations of provided annotation class
   *
   * @param annotation class
   * @return optional annotation
   */
  <T extends TypeAnnotation> Optional<T> getAttributesAnnotation(Class<T> annotation);

  /**
   * Creates a builder for the MessageMetadataType based on the specified message
   *
   * @param messageType The message type to build builder from.
   * @return The builder
   */
  static MessageMetadataTypeBuilder builder(ObjectType messageType) {
    return MessageMetadataTypeBuilder.fromObjectType(messageType);
  }

}
