/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.message.api;


import static java.util.Collections.addAll;
import static java.util.stream.Stream.of;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.BaseMetadataType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * Default implementation for {@link MessageMetadataType}.
 *
 * @since 1.0
 */
class DefaultMessageMetadataType extends BaseMetadataType implements ObjectType, MessageMetadataType {

  public static final String PAYLOAD_FIELD_NAME = "payload";
  public static final String ATTRIBUTES_FIELD_NAME = "attributes";
  private final Optional<ObjectFieldType> payloadMetadataType;
  private final Optional<ObjectFieldType> attributes;
  private Object[] fieldValues;


  DefaultMessageMetadataType(Optional<MetadataType> payloadMetadataType, Optional<MetadataType> attributes,
                             Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    this(payloadMetadataType, null, attributes, null, annotations);
  }

  DefaultMessageMetadataType(Optional<MetadataType> payloadMetadataType,
                             Consumer<ObjectFieldTypeBuilder> payloadObjectFieldTypeBuilderConsumer,
                             Optional<MetadataType> attributes,
                             Consumer<ObjectFieldTypeBuilder> attributesObjectFieldTypeBuilderConsumer,
                             Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    super(MetadataFormat.JAVA, annotations);
    this.payloadMetadataType = payloadMetadataType.map((type) -> {
      final ObjectFieldTypeBuilder objectFieldTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
      objectFieldTypeBuilder.key(PAYLOAD_FIELD_NAME).required(true).value(type);
      Optional.ofNullable(payloadObjectFieldTypeBuilderConsumer)
          .ifPresent(objectFieldTypeBuilderConsumer -> payloadObjectFieldTypeBuilderConsumer.accept(objectFieldTypeBuilder));
      return objectFieldTypeBuilder.build();
    });

    this.attributes = attributes.map((type) -> {
      final ObjectFieldTypeBuilder objectFieldTypeBuilder = new ObjectFieldTypeBuilder(MetadataFormat.JAVA);
      objectFieldTypeBuilder.key(ATTRIBUTES_FIELD_NAME).required(true).value(type);
      Optional.ofNullable(payloadObjectFieldTypeBuilderConsumer)
          .ifPresent(objectFieldTypeBuilderConsumer -> attributesObjectFieldTypeBuilderConsumer.accept(objectFieldTypeBuilder));
      return objectFieldTypeBuilder.build();
    });
  }

  @Override
  public Optional<MetadataType> getOpenRestriction() {
    return Optional.empty();
  }

  @Override
  public Collection<ObjectFieldType> getFields() {
    return of(payloadMetadataType, attributes)
        .filter((t) -> t.isPresent())
        .map((t) -> t.get())
        .collect(Collectors.toList());
  }

  @Override
  public boolean isOrdered() {
    return false;
  }

  @Override
  public Optional<ObjectFieldType> getFieldByName(String name) {
    switch (name) {
      case PAYLOAD_FIELD_NAME:
        return payloadMetadataType;
      case ATTRIBUTES_FIELD_NAME:
        return attributes;
    }
    return Optional.empty();
  }

  @Override
  public void accept(MetadataTypeVisitor metadataTypeVisitor) {
    metadataTypeVisitor.visitObject(this);
  }

  @Override
  public Optional<MetadataType> getPayloadType() {
    return payloadMetadataType.map(ObjectFieldType::getValue);
  }

  @Override
  public Optional<Set<TypeAnnotation>> getPayloadAnnotations() {
    return payloadMetadataType.map(MetadataType::getAnnotations);
  }

  @Override
  public <T extends TypeAnnotation> Optional<T> getPayloadAnnotation(Class<T> annotation) {
    return payloadMetadataType.flatMap((t) -> t.getAnnotation(annotation));
  }

  @Override
  public Optional<MetadataType> getAttributesType() {
    return attributes.map(ObjectFieldType::getValue);
  }

  @Override
  public Optional<Set<TypeAnnotation>> getAttributesAnnotations() {
    return attributes.map(MetadataType::getAnnotations);
  }

  @Override
  public <T extends TypeAnnotation> Optional<T> getAttributesAnnotation(Class<T> annotation) {
    return attributes.flatMap((t) -> t.getAnnotation(annotation));
  }

  public Object[] getFieldValues() {
    if (fieldValues == null) {
      fieldValues = createFieldValuesArray();
    }
    return fieldValues;
  }

  private Object[] createFieldValuesArray() {
    List<Object> fieldValues = new ArrayList<>();
    addAll(fieldValues, super.getFieldValues());
    addAll(fieldValues, payloadMetadataType, attributes);
    return fieldValues.toArray(new Object[fieldValues.size()]);
  }
}
