/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.Accessibility;
import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.FieldOccurrenceAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.impl.DefaultObjectFieldType;
import org.mule.metadata.internal.utils.LazyValue;

import javax.xml.namespace.QName;
import java.util.Optional;
import java.util.regex.Pattern;

public class ObjectFieldTypeBuilder extends AbstractBuilder<ObjectFieldType>
    implements TypeBuilder<ObjectFieldType>, WithAnnotation<ObjectFieldTypeBuilder> {

  private Optional<ObjectKeyBuilder> keyBuilder;
  private Optional<? extends TypeBuilder<?>> valueBuilder;
  private boolean required;
  private boolean repeated;

  private LazyValue<ObjectFieldType> result;
  private MetadataType typeValue;

  public ObjectFieldTypeBuilder(MetadataFormat format) {
    super(format);
    this.keyBuilder = Optional.empty();
    this.valueBuilder = Optional.empty();
    this.required = false;
    this.repeated = false;
    this.result = new LazyValue<>();
  }

  public ObjectFieldTypeBuilder with(TypeAnnotation extension) {
    addExtension(extension);
    return this;
  }

  public ObjectFieldTypeBuilder withKeyAnnotation(TypeAnnotation extension) {
    keyBuilder.ifPresent(builder -> builder.addExtension(extension));
    return this;
  }

  public ObjectFieldTypeBuilder description(String lang, String content) {
    return with(new DescriptionAnnotation(content, lang));
  }

  public ObjectFieldTypeBuilder description(String content) {
    return with(new DescriptionAnnotation(content));
  }

  public ObjectFieldTypeBuilder required() {
    return required(true);
  }

  public ObjectFieldTypeBuilder required(boolean required) {
    this.required = required;
    return this;
  }

  public ObjectFieldTypeBuilder repeated(boolean repeated) {
    this.repeated = repeated;
    return this;
  }

  public ObjectFieldTypeBuilder accessibility(Accessibility accessibility) {
    return with(new AccessibilityAnnotation(accessibility));
  }

  public ObjectFieldTypeBuilder occurrence(Number min, Number max) {
    return with(new FieldOccurrenceAnnotation(min, max));
  }

  public ObjectFieldTypeBuilder label(String label) {
    withKeyAnnotation(new LabelAnnotation(label));
    return this;
  }

  public ObjectKeyBuilder key(QName keyName) {
    final ObjectKeyBuilder keyBuilder = new ObjectKeyBuilder(format);
    keyBuilder.name(keyName);
    this.keyBuilder = Optional.of(keyBuilder);
    return keyBuilder;
  }

  public ObjectFieldTypeBuilder key(String keyName) {
    final ObjectKeyBuilder keyBuilder = new ObjectKeyBuilder(format);
    keyBuilder.name(keyName);
    this.keyBuilder = Optional.of(keyBuilder);
    return this;
  }

  public ObjectFieldTypeBuilder key(Pattern pattern) {
    final ObjectKeyBuilder keyBuilder = new ObjectKeyBuilder(format);
    keyBuilder.pattern(pattern);
    this.keyBuilder = Optional.of(keyBuilder);
    return this;
  }

  public BaseTypeBuilder value() {
    final Optional<BaseTypeBuilder> typeBuilder = Optional.of(new BaseTypeBuilder(format));
    valueBuilder = typeBuilder;
    return typeBuilder.get();
  }


  public ObjectFieldTypeBuilder value(MetadataType typeValue) {
    this.typeValue = typeValue;
    return this;
  }

  public ObjectFieldTypeBuilder value(TypeBuilder<?> builder) {
    valueBuilder = Optional.of(builder);
    return this;
  }

  @Override
  public ObjectFieldType build() {
    return result.get(() -> {
      final ObjectKeyType keyType =
          keyBuilder.orElseThrow(() -> new RuntimeException("No field key was specified for building the ObjectType.")).build();
      final MetadataType value = typeValue != null ? typeValue
          : valueBuilder.orElseThrow(() -> new RuntimeException("No field value was specified for building the ObjectType."))
              .build();
      return new DefaultObjectFieldType(keyType, value, required, repeated, format, annotations);
    });
  }

}
