/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.utils;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.WithAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * Utilities for handling values
 *
 * @since 1.0
 */
public final class MetadataTypeUtils {

  private MetadataTypeUtils() {}

  /**
   * @param condition Condition that the argument must satisfy
   * @param message   The Message of the exception in case the condition is invalid
   */
  public static void checkArgument(boolean condition, String message) {
    if (!condition) {
      throw new IllegalArgumentException(message);
    }
  }

  /**
   * Returns the type id of a {@link MetadataType} if exist.
   *
   * @param type the metadata type to search de type id from.
   */
  public static Optional<String> getTypeId(MetadataType type) {
    TypeResolverVisitor visitor = new TypeResolverVisitor();
    type.accept(visitor);
    return Optional.ofNullable(visitor.getResolvedTypeId());
  }

  /**
   * Gives a type alias to type specified by builder.
   *
   * @param typeBuilder The type builder to annotate.
   * @param typeAlias The type alias to use.
   */
  public static void addTypeAlias(TypeBuilder typeBuilder, String typeAlias) {
    if (typeAlias == null || typeBuilder == null) {
      return;
    }

    if (typeBuilder instanceof WithAnnotation) {
      final WithAnnotation withAnnotation = (WithAnnotation) typeBuilder;
      withAnnotation.with(new TypeAliasAnnotation(typeAlias));
    }
  }

  /**
   * Returns the default value of a {@link MetadataType} if exist.
   *
   * @param type the metadata type to search de default value from.
   */
  public static Optional<String> getDefaultValue(MetadataType type) {
    return type.getAnnotation(DefaultValueAnnotation.class).map(DefaultValueAnnotation::getValue);
  }

  /**
   * Returns whether a metadata type is an instance of {@link VoidType} or not.
   *
   * @param type the metadata type to check.
   */
  public static boolean isVoid(MetadataType type) {
    return type instanceof VoidType;
  }

  /**
   * Returns whether a metadata type is an instance of {@link ObjectType} or not.
   *
   * @param type the metadata type to check.
   */
  public static boolean isObjectType(MetadataType type) {
    return type instanceof ObjectType;
  }

  /**
   * Returns if an {@link ObjectType} has at least one exposed field or not, for any other {@link MetadataType}
   * returns false.
   *
   * @param type the metadata type to check.
   */
  public static boolean hasExposedFields(MetadataType type) {
    return type instanceof ObjectType && !((ObjectType) type).getFields().isEmpty();
  }

  /**
   * Returns the local part of an {@link ObjectFieldType}
   *
   * @param field the field to obtain the local part
   */
  public static String getLocalPart(ObjectFieldType field) {
    return field.getKey().getName().getLocalPart();
  }

  /**
   * Indicates whether the give {@link MetadataType} is an Enum or not.
   *
   * @param type {@link MetadataType} to check if it is an Enum
   */
  public static boolean isEnum(MetadataType type) {
    return type.getAnnotation(EnumAnnotation.class).isPresent();
  }

  public static int hashCode(Optional<?>... optionals) {
    HashCodeBuilder hashCodeBuilder = new HashCodeBuilder();
    Stream.of(optionals)
        .filter(Optional::isPresent)
        .forEach(n -> hashCodeBuilder.append(n.get()));

    return hashCodeBuilder.toHashCode();
  }

  /**
   * Checks if the given object is null or not
   */
  public static boolean isNotNull(Object object) {
    return object != null;
  }

  /**
   * {@link MetadataTypeVisitor} implementation for resolving the typeId of a given {@link MetadataType}.
   */
  private static class TypeResolverVisitor extends MetadataTypeVisitor {

    private String typeId = null;

    @Override
    public void visitString(StringType stringType) {
      Optional<EnumAnnotation> enumAnnotation = stringType.getAnnotation(EnumAnnotation.class);
      if (enumAnnotation.isPresent()) {
        defaultVisit(stringType);
      } else {
        typeId = String.class.getName();
      }
    }

    @Override
    public void visitObjectField(ObjectFieldType objectFieldType) {
      defaultVisit(objectFieldType.getValue());
    }

    @Override
    public void visitAnyType(AnyType anyType) {
      typeId = Object.class.getName();
    }

    @Override
    public void defaultVisit(MetadataType metadataType) {
      typeId = metadataType.getAnnotation(TypeIdAnnotation.class).map(TypeIdAnnotation::getValue).orElse(null);
    }

    private String getResolvedTypeId() {
      return typeId;
    }
  }

}
