/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.FunctionParameter;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultFunctionType;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public class FunctionTypeBuilder extends AbstractBuilder<FunctionType>
    implements TypeBuilder<FunctionType>, WithAnnotation<FunctionTypeBuilder> {

  private List<FunctionParameterBuilder> parameterBuilders = new ArrayList<>();
  private MetadataType returnType;
  private Optional<TypeBuilder<?>> returnTypeBuilder = Optional.empty();

  protected FunctionTypeBuilder(MetadataFormat format) {
    super(format);
  }

  public FunctionTypeBuilder with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  public FunctionTypeBuilder id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public FunctionTypeBuilder defaultValue(String defaultValue) {
    return with(new DefaultValueAnnotation(defaultValue));
  }

  public BaseTypeBuilder addParameterOf(String name) {
    final BaseTypeBuilder typeBuilder = new BaseTypeBuilder(format);
    this.parameterBuilders.add(new FunctionParameterBuilder(name, typeBuilder));
    return typeBuilder;
  }

  public BaseTypeBuilder addOptionalParameterOf(String name) {
    final BaseTypeBuilder typeBuilder = new BaseTypeBuilder(format);
    this.parameterBuilders.add(new FunctionParameterBuilder(name, typeBuilder, true));
    return typeBuilder;
  }

  public FunctionTypeBuilder addParameterOf(String name, TypeBuilder<?> paramType) {
    this.parameterBuilders.add(new FunctionParameterBuilder(name, paramType));
    return this;
  }

  public FunctionTypeBuilder addOptionalParameterOf(String name, TypeBuilder<?> paramType) {
    this.parameterBuilders.add(new FunctionParameterBuilder(name, paramType, true));
    return this;
  }

  public FunctionTypeBuilder addParameterOf(String name, MetadataType paramType) {
    this.parameterBuilders.add(new FunctionParameterBuilder(name, paramType));
    return this;
  }

  public FunctionTypeBuilder addOptionalParameterOf(String name, MetadataType paramType) {
    this.parameterBuilders.add(new FunctionParameterBuilder(name, paramType, true));
    return this;
  }

  public FunctionTypeBuilder returnType(TypeBuilder<?> returnTypeBuilder) {
    this.returnTypeBuilder = Optional.of(returnTypeBuilder);
    return this;
  }

  public FunctionTypeBuilder returnType(MetadataType returnType) {
    this.returnType = returnType;
    return this;
  }

  public BaseTypeBuilder returnType() {
    final BaseTypeBuilder typeBuilder = new BaseTypeBuilder(format);
    this.returnTypeBuilder = Optional.of(typeBuilder);
    return typeBuilder;
  }

  @Override
  public FunctionType build() {
    Optional<MetadataType> returnTypeOptional =
        returnType != null ? Optional.of(returnType) : returnTypeBuilder.map(TypeBuilder::build);
    return new DefaultFunctionType(format,
                                   annotations,
                                   returnTypeOptional,
                                   parameterBuilders.stream().map((ptb) -> ptb.build()).collect(Collectors.toList()));
  }

  private class FunctionParameterBuilder {

    private String name;
    private MetadataType metadataType;
    private boolean optional;
    private TypeBuilder<?> builder;

    FunctionParameterBuilder(String name, TypeBuilder<?> builder) {
      this(name, builder, false);
    }

    FunctionParameterBuilder(String name, TypeBuilder<?> builder, boolean optional) {
      this.name = name;
      this.builder = builder;
      this.optional = optional;
    }

    FunctionParameterBuilder(String name, MetadataType metadataType) {
      this(name, metadataType, false);
    }

    FunctionParameterBuilder(String name, MetadataType metadataType, boolean optional) {
      this.name = name;
      this.metadataType = metadataType;
      this.optional = optional;
    }

    private FunctionParameter build() {
      MetadataType type = metadataType != null ? metadataType : builder.build();
      return new FunctionParameter(name, type, optional);
    }
  }
}
