/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultAttributeFieldType;
import org.mule.metadata.api.model.impl.DefaultAttributeKeyType;

import javax.xml.namespace.QName;
import java.util.Collections;
import java.util.Optional;
import java.util.regex.Pattern;

public class AttributeFieldTypeBuilder extends AbstractBuilder<AttributeFieldType>
    implements TypeBuilder<AttributeFieldType> {

  private Optional<QName> name;
  private Optional<Pattern> pattern;
  private Optional<TypeBuilder<?>> value;
  private boolean required;

  protected AttributeFieldTypeBuilder(MetadataFormat format) {
    super(format);
    this.name = Optional.empty();
    this.pattern = Optional.empty();
    this.value = Optional.empty();
    this.required = false;
  }

  public AttributeFieldTypeBuilder name(QName name) {
    this.name = Optional.of(name);
    return this;
  }

  public AttributeFieldTypeBuilder name(String name) {
    this.name = Optional.of(new QName(name));
    return this;
  }

  public AttributeFieldTypeBuilder pattern(Pattern pattern) {
    this.pattern = Optional.of(pattern);
    return this;
  }

  public BaseTypeBuilder value() {
    final BaseTypeBuilder typeBuilder = new BaseTypeBuilder(format);
    this.value = Optional.of(typeBuilder);
    return typeBuilder;
  }

  public AttributeFieldTypeBuilder value(TypeBuilder baseTypeBuilder) {
    this.value = Optional.of(baseTypeBuilder);
    return this;
  }

  public void required(boolean required) {
    this.required = required;
  }

  @Override
  public AttributeFieldType build() {
    if (!name.isPresent() && !pattern.isPresent()) {
      throw new RuntimeException("No name or pattern was specified.");
    }
    final TypeBuilder<?> typeBuilder = this.value.orElseThrow(() -> new RuntimeException("No value was specified."));
    final MetadataType build = typeBuilder.build();
    final DefaultAttributeKeyType key = new DefaultAttributeKeyType(name, pattern, format, Collections.emptyMap());
    return new DefaultAttributeFieldType(key, build, required, format, annotations);
  }

}
