/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static java.util.Collections.unmodifiableMap;
import static org.apache.commons.lang3.builder.ToStringBuilder.reflectionToString;
import static org.apache.commons.lang3.builder.ToStringStyle.MULTI_LINE_STYLE;
import static org.mule.metadata.internal.utils.EfficientReflectionEquals.reflectionEquals;
import static org.mule.metadata.internal.utils.EfficientReflectionHashCode.reflectionHashCode;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ReflectionAwareComparable;

import java.lang.reflect.Field;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public abstract class BaseMetadataType implements MetadataType, ReflectionAwareComparable {

  protected final Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations;
  private final MetadataFormat metadataFormat;

  public BaseMetadataType(MetadataFormat metadataFormat, Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    this.metadataFormat = metadataFormat;
    this.annotations = annotations == null ? null : unmodifiableMap(annotations);
  }

  protected static final Field[] BASE_REFLECTION_FIELDS;
  static {
    try {
      BASE_REFLECTION_FIELDS = new Field[] {
          BaseMetadataType.class.getDeclaredField("metadataFormat"),
          BaseMetadataType.class.getDeclaredField("annotations")
      };
    } catch (NoSuchFieldException e) {
      throw new RuntimeException("Could not resolve field", e);
    }
  }

  @Override
  public Set<TypeAnnotation> getAnnotations() {
    return new LinkedHashSet<>(annotations.values());
  }

  @Override
  public MetadataFormat getMetadataFormat() {
    return metadataFormat;
  }

  @Override
  public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> extension) {
    return Optional.ofNullable((T) annotations.get(extension));
  }

  @Override
  public Optional<String> getDescription() {
    return getAnnotation(DescriptionAnnotation.class).map(DescriptionAnnotation::getValue);
  }

  public Field[] getReflectionComparableFields() {
    return BASE_REFLECTION_FIELDS;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || getClass() != obj.getClass()) {
      return false;
    }

    return reflectionEquals(this, (BaseMetadataType) obj);
  }

  private transient boolean hashCalculated;
  private transient int hash;

  @Override
  public int hashCode() {
    if (!hashCalculated) {
      hash = reflectionHashCode(this);
      hashCalculated = true;
    }
    return hash;
  }

  @Override
  public String toString() {
    return reflectionToString(this, MULTI_LINE_STYLE);
  }
}
