/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.internal.utils.LazyValue;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class ObjectTypeBuilder extends AbstractBuilder<ObjectType>
    implements TypeBuilder<ObjectType>, WithAnnotation<ObjectTypeBuilder> {

  private List<ObjectFieldTypeBuilder> fieldsBuilder;
  private boolean ordered;
  private Optional<TypeBuilder<?>> openRestriction;
  private Optional<MetadataType> openRestrictionType;
  private LazyValue<ObjectType> result;

  public ObjectTypeBuilder(MetadataFormat format) {
    super(format);
    this.fieldsBuilder = new ArrayList<>();
    this.result = new LazyValue<>();
    this.openRestriction = empty();
    this.openRestrictionType = empty();
    this.ordered = false;
  }

  public ObjectFieldTypeBuilder addField() {
    final ObjectFieldTypeBuilder fieldMetadataBuilder = new ObjectFieldTypeBuilder(format);
    this.fieldsBuilder.add(fieldMetadataBuilder);
    return fieldMetadataBuilder;
  }

  public ObjectTypeBuilder ordered(boolean ordered) {
    this.ordered = ordered;
    return this;
  }

  public ObjectTypeBuilder id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public ObjectTypeBuilder with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  public ObjectTypeBuilder description(String lang, String content) {
    return with(new DescriptionAnnotation(content, lang));
  }

  public ObjectTypeBuilder description(String content) {
    return with(new DescriptionAnnotation(content));
  }

  public ObjectTypeBuilder label(String label) {
    return with(new LabelAnnotation(label));
  }

  public ObjectTypeBuilder open() {
    this.openRestriction = Optional.of(new AnyTypeBuilder(format));
    return this;
  }

  public BaseTypeBuilder openWith() {
    BaseTypeBuilder baseTypeBuilder = BaseTypeBuilder.create(format);
    this.openRestriction = Optional.of(baseTypeBuilder);
    return baseTypeBuilder;
  }

  public ObjectTypeBuilder openWith(TypeBuilder openRestriction) {
    this.openRestriction = Optional.of(openRestriction);
    return this;
  }

  public ObjectTypeBuilder openWith(MetadataType openRestriction) {
    this.openRestrictionType = Optional.of(openRestriction);
    return this;
  }

  @Override
  public ObjectType build() {
    final ArrayList<ObjectFieldType> fields = new ArrayList<>();
    // This is to avoid stack overflow we create the object with no fields
    boolean needsInit = !result.isDefined();
    final ObjectType objectType = result.get(() -> {
      MetadataType restrictionType = openRestriction.map(TypeBuilder::build).orElse(null);
      restrictionType = restrictionType == null ? openRestrictionType.orElse(null) : restrictionType;
      return new DefaultObjectType(fields, ordered, restrictionType, format, annotations);
    });
    if (needsInit) {
      final List<ObjectFieldType> collect = fieldsBuilder.stream()
          .map((fieldBuilder) -> fieldBuilder.build())
          .collect(toList());
      fields.addAll(collect);
    }
    return objectType;
  }

}
