/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static org.mule.metadata.internal.utils.EfficientEquals.efficientEquals;
import static org.mule.metadata.internal.utils.EfficientHashCode.efficientHashcode;

import static java.util.Collections.unmodifiableMap;

import static org.apache.commons.lang3.builder.ToStringStyle.MULTI_LINE_STYLE;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.FieldsComparable;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.ToStringBuilder;

public abstract class BaseMetadataType implements MetadataType, FieldsComparable {

  protected final Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations;
  private final MetadataFormat metadataFormat;

  public BaseMetadataType(MetadataFormat metadataFormat, Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    this.metadataFormat = metadataFormat;
    this.annotations = annotations == null ? null : unmodifiableMap(annotations);
  }

  @Override
  public Set<TypeAnnotation> getAnnotations() {
    return new LinkedHashSet<>(annotations.values());
  }

  @Override
  public MetadataFormat getMetadataFormat() {
    return metadataFormat;
  }

  @Override
  public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> extension) {
    return Optional.ofNullable((T) annotations.get(extension));
  }

  @Override
  public Optional<String> getDescription() {
    return getAnnotation(DescriptionAnnotation.class).map(DescriptionAnnotation::getValue);
  }

  public Object[] getFieldValues() {
    return new Object[] {annotations, metadataFormat};
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || getClass() != obj.getClass()) {
      return false;
    }

    return efficientEquals(this, (BaseMetadataType) obj);
  }

  private transient boolean hashCalculated;
  private transient int hash;

  @Override
  public int hashCode() {
    if (!hashCalculated) {
      hash = efficientHashcode(this);
      hashCalculated = true;
    }
    return hash;
  }

  @Override
  public String toString() {
    // This is done to remove reflection in the toString.
    ToStringBuilder toStringBuilder = new ToStringBuilder(this, MULTI_LINE_STYLE);
    for (Object fieldValue : getFieldValues()) {
      toStringBuilder.append(fieldValue);
    }
    return toStringBuilder.toString();
  }
}
