/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static org.mule.metadata.internal.utils.EfficientEquals.efficientEquals;
import static org.mule.metadata.internal.utils.EfficientHashCode.efficientHashcode;

import static java.util.Collections.addAll;
import static java.util.Objects.hash;

import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;
import javax.xml.namespace.QName;

public class DefaultObjectKeyType extends BaseMetadataType implements ObjectKeyType {

  private final Optional<QName> name;
  private final Optional<Pattern> pattern;
  private final Collection<AttributeFieldType> attributes;
  private Object[] fieldValues;

  public DefaultObjectKeyType(Optional<QName> name, Optional<Pattern> pattern, Collection<AttributeFieldType> attributes,
                              MetadataFormat metadataFormat, Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions) {
    super(metadataFormat, extensions);
    this.name = name;
    this.pattern = pattern;
    this.attributes = attributes;
  }

  @Override
  public QName getName() {
    return name.get();
  }

  @Override
  public boolean isName() {
    return name.isPresent();
  }

  @Override
  public Pattern getPattern() {
    return pattern.get();
  }

  @Override
  public boolean isPattern() {
    return pattern.isPresent();
  }

  public Optional<String> getLabel() {
    return getAnnotation(LabelAnnotation.class).map((label) -> label.getValue());
  }

  @Override
  public Collection<AttributeFieldType> getAttributes() {
    return attributes;
  }

  @Override
  public void accept(MetadataTypeVisitor visitor) {
    visitor.visitObjectKey(this);
  }

  @Override
  public String toString() {
    return "DefaultObjectKeyType{name: " + name.toString() + "}";
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof DefaultObjectKeyType)) {
      return false;
    }

    DefaultObjectKeyType that = (DefaultObjectKeyType) o;
    return efficientEquals(this, that) && equalPattern(pattern, that.pattern);
  }

  boolean equalPattern(Optional<Pattern> thisPattern, Optional<Pattern> thatPattern) {
    if (thisPattern.isPresent() != thatPattern.isPresent()) {
      return false;
    }

    if (!thisPattern.isPresent() || !thatPattern.isPresent()) {
      return true;
    }

    return thisPattern.get().toString().equals(thatPattern.get().toString());
  }

  private transient boolean hashCalculated;
  private transient int hash;

  @Override
  public int hashCode() {
    if (!hashCalculated) {
      hash = hash(efficientHashcode(this), pattern.map(x -> x.toString().hashCode()).orElse(null));
      hashCalculated = true;
    }
    return hash;
  }

  @Override
  public Object[] getFieldValues() {
    if (fieldValues == null) {
      fieldValues = createFieldValuesArray();
    }

    return fieldValues;
  }

  private Object[] createFieldValuesArray() {
    List<Object> fieldValues = new ArrayList<>();
    addAll(fieldValues, super.getFieldValues());
    addAll(fieldValues, attributes, name);
    return fieldValues.toArray(new Object[fieldValues.size()]);
  }
}
