/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.internal.utils;

import org.mule.metadata.api.model.FieldsComparable;

import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;

/**
 * A more efficient version of {@link org.apache.commons.lang3.builder.EqualsBuilder}. Extracts the fields of the classes to check
 * via the {@link FieldsComparable} class.
 *
 * @since 1.3
 */
public class EfficientEquals {

  private static final ThreadLocal<Set<Pair<IDKey, IDKey>>> EQUALS_REGISTRY = new ThreadLocal<>();

  static Set<Pair<IDKey, IDKey>> getRegistry() {
    return EQUALS_REGISTRY.get();
  }

  static Pair<IDKey, IDKey> getRegisterPair(final Object lhs, final Object rhs) {
    final IDKey left = new IDKey(lhs);
    final IDKey right = new IDKey(rhs);
    return Pair.of(left, right);
  }

  static boolean isRegistered(final Object lhs, final Object rhs) {
    final Set<Pair<IDKey, IDKey>> registry = getRegistry();
    final Pair<IDKey, IDKey> pair = getRegisterPair(lhs, rhs);
    final Pair<IDKey, IDKey> swappedPair = Pair.of(pair.getRight(), pair.getLeft());

    return registry != null
        && (registry.contains(pair) || registry.contains(swappedPair));
  }

  private static void register(final Object lhs, final Object rhs) {
    Set<Pair<IDKey, IDKey>> registry = getRegistry();
    if (registry == null) {
      registry = new HashSet<>();
      EQUALS_REGISTRY.set(registry);
    }
    final Pair<IDKey, IDKey> pair = getRegisterPair(lhs, rhs);
    registry.add(pair);
  }

  private static void unregister(final Object lhs, final Object rhs) {
    final Set<Pair<IDKey, IDKey>> registry = getRegistry();
    if (registry != null) {
      final Pair<IDKey, IDKey> pair = getRegisterPair(lhs, rhs);
      registry.remove(pair);
      if (registry.isEmpty()) {
        EQUALS_REGISTRY.remove();
      }
    }
  }

  private EfficientEquals() {}

  public static boolean efficientEquals(FieldsComparable thisObj, FieldsComparable thatObj) {
    if (isRegistered(thisObj, thatObj)) {
      return true;
    }

    try {
      register(thisObj, thatObj);

      Object[] theseFields = thisObj.getFieldValues();
      Object[] thoseFields = thatObj.getFieldValues();

      for (int i = 0; i < thoseFields.length; i++) {
        Object thisField = theseFields[i];
        Object thatField = thoseFields[i];

        if (thisField == thatField) {
          continue;
        }

        if (thisField == null) {
          return false;
        }

        if (!thisField.equals(thatField)) {
          return false;
        }
      }

      return true;
    } finally {
      unregister(thisObj, thatObj);
    }
  }
}
