/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model;

import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Objects;
import java.util.Optional;

/**
 * Represent Data Format like Xml, Json, CSV, Java etc
 */
public class MetadataFormat {

  private final String label;
  private final String id;
  private final Collection<String> validMimeTypes;

  public static MetadataFormat JAVA = new MetadataFormat("Java", "java", "application/java");
  public static MetadataFormat JSON = new MetadataFormat("Json", "json", "application/json");
  public static MetadataFormat XML = new MetadataFormat("XML", "xml", "application/xml");
  public static MetadataFormat CSV = new MetadataFormat("CSV", "csv", "application/csv");

  public MetadataFormat(String label, String id, String... validMimeTypes) {
    if (StringUtils.isBlank(id)) {
      throw new IllegalArgumentException("id cannot be null");
    }

    this.id = id;
    this.label = label;
    this.validMimeTypes = validMimeTypes != null
        ? Collections.unmodifiableList(Arrays.asList(validMimeTypes))
        : Collections.emptyList();
  }

  public Optional<String> getLabel() {
    return Optional.ofNullable(label);
  }

  public Collection<String> getValidMimeTypes() {
    return validMimeTypes;
  }

  public String getId() {
    return id;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof MetadataFormat) {
      return Objects.equals(id, ((MetadataFormat) obj).id);
    }

    return false;
  }

  @Override
  public int hashCode() {
    return id.hashCode();
  }
}
