/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.*;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.api.model.impl.DefaultUnionType;
import org.mule.metadata.internal.utils.LazyValue;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

public class UnionTypeBuilder extends AbstractBuilder<UnionType>
    implements TypeBuilder<UnionType>, WithAnnotation<UnionTypeBuilder> {

  private final List<Supplier<MetadataType>> types = new LinkedList<>();

  private final LazyValue<UnionType> result = new LazyValue<>();


  protected UnionTypeBuilder(MetadataFormat format) {
    super(format);
  }

  public BaseTypeBuilder of() {
    final BaseTypeBuilder builder = new BaseTypeBuilder(format);
    of(builder);
    return builder;
  }

  public UnionTypeBuilder of(TypeBuilder<?> builder) {
    this.types.add(builder::build);
    return this;
  }

  public UnionTypeBuilder of(MetadataType type) {
    this.types.add(() -> type);
    return this;
  }

  public UnionTypeBuilder id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public UnionTypeBuilder with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  @Override
  public UnionType build() {
    if (types.isEmpty()) {
      throw new RuntimeException("Union needs at least one type");
    }
    final ArrayList<MetadataType> types = new ArrayList<>();
    // This is to avoid stack overflow we create the object with no fields
    boolean needsInit = !result.isDefined();
    final UnionType unionType = result.get(() -> new DefaultUnionType(types, format, annotations));
    if (needsInit) {
      final List<MetadataType> createdTypes = this.types.stream().map(Supplier::get).collect(Collectors.toList());
      types.addAll(createdTypes);
    }
    return unionType;
  }
}
