/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.internal.utils;

import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;

public class IoUtils {

  /**
   * Returns an {@link InputStream} that will read from an {@link URL} connection without caching when the resource is inside a
   * jar. This is important when working with resources obtained via {@link ClassLoader#getResource(String)} in order to avoid
   * file descriptor leaks (closing the stream does not clean the cache).
   * <p>
   * Note that {@link ClassLoader#getResourceAsStream(String)} already take care of closing such resources as part of closing the
   * stream, therefore caching is not a problem in that case.
   *
   * @param url The URL to connect to.
   * @return The InputStream that will not leave any cached resource after closing.
   * @throws IOException If it fails while obtaining the InputStream.
   */
  public static InputStream getInputStreamWithCacheControl(URL url) throws IOException {
    URLConnection urlConnection = url.openConnection();
    if (urlConnection instanceof JarURLConnection) {
      urlConnection.setUseCaches(false);
    }
    return urlConnection.getInputStream();
  }
}
