/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ReferenceResolver;
import org.mule.metadata.api.model.ReferenceType;
import org.mule.metadata.api.model.impl.DefaultReferenceType;

/**
 * Builder to create a Reference Type
 *
 * @since 1.8.0
 */
public class ReferenceTypeBuilder extends AbstractBuilder<ReferenceType>
    implements TypeBuilder<ReferenceType>, WithAnnotation<ReferenceTypeBuilder> {


  private ReferenceResolver resolver;
  private String name;


  protected ReferenceTypeBuilder(MetadataFormat format) {
    super(format);
  }

  /**
   * Specifies the resolver to be used. This resolver is going to be called when resolving the reference is needed
   *
   * @param resolver The resolver
   * @return The builder with the resolver being set
   */
  public ReferenceTypeBuilder withResolver(ReferenceResolver resolver) {
    this.resolver = resolver;
    return this;
  }

  /**
   * The name of the type to be resolved. This name must be a unique identifier of the type being referenced
   *
   * @param name The name of the type
   * @return The builder with the name being set
   */
  public ReferenceTypeBuilder withName(String name) {
    this.name = name;
    return this;
  }

  @Override
  public ReferenceType build() {
    if (resolver == null) {
      throw new IllegalStateException("Resolver can not be null, use withResolver to specify the type resolver.");
    }

    if (name == null) {
      throw new IllegalStateException("Name can not be null, use withName to specify the type name.");
    }

    return new DefaultReferenceType(resolver, name, format, annotations);
  }

  @Override
  public ReferenceTypeBuilder with(TypeAnnotation typeAnnotation) {
    this.addExtension(typeAnnotation);
    return this;
  }
}
