/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.internal.utils;

import static java.lang.Double.doubleToLongBits;

import org.mule.metadata.api.model.FieldsComparable;

import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang3.builder.Builder;


/**
 * A more efficient version of {@link org.apache.commons.lang3.builder.HashCodeBuilder#reflectionHashCode(Object, boolean)}.
 * Extracts the fields of the classes to hash via the {@link FieldsComparable} class when possible.
 *
 * @since 1.3
 */
public class EfficientHashCode implements Builder<Integer> {

  private static final int DEFAULT_INITIAL_VALUE = 17;

  private static final int DEFAULT_MULTIPLIER_VALUE = 37;

  private static final ThreadLocal<Set<IDKey>> REGISTRY = new ThreadLocal<>();

  static Set<IDKey> getRegistry() {
    return REGISTRY.get();
  }

  static boolean isRegistered(final Object value) {
    final Set<IDKey> registry = getRegistry();
    return registry != null && registry.contains(new IDKey(value));
  }

  private static void efficientAppend(final Object object, final EfficientHashCode builder) {
    if (isRegistered(object)) {
      return;
    }

    try {
      register(object);

      if (object instanceof FieldsComparable) {
        Object[] fields = ((FieldsComparable) object).getFieldValues();

        for (Object field : fields) {
          efficientAppend(field, builder);
        }
      } else if (object != null) {
        builder.append(object);
      }
    } finally {
      unregister(object);
    }
  }


  public static int efficientHashcode(FieldsComparable object) {
    final EfficientHashCode builder = new EfficientHashCode();

    efficientAppend(object, builder);

    return builder.build();
  }

  private static void register(final Object value) {
    Set<IDKey> registry = getRegistry();
    if (registry == null) {
      registry = new HashSet<>();
      REGISTRY.set(registry);
    }
    registry.add(new IDKey(value));
  }

  private static void unregister(final Object value) {
    final Set<IDKey> registry = getRegistry();
    if (registry != null) {
      registry.remove(new IDKey(value));
      if (registry.isEmpty()) {
        REGISTRY.remove();
      }
    }
  }

  /**
   * Constant to use in building the hashCode.
   */
  private final int iConstant;

  /**
   * Running total of the hashCode.
   */
  private int iTotal;

  public EfficientHashCode() {
    iConstant = DEFAULT_MULTIPLIER_VALUE;
    iTotal = DEFAULT_INITIAL_VALUE;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code boolean}.
   * </p>
   * <p>
   * This adds {@code 1} when true, and {@code 0} when false to the {@code hashCode}.
   * </p>
   * <p>
   * This is in contrast to the standard {@code java.lang.Boolean.hashCode} handling, which computes a {@code hashCode} value of
   * {@code 1231} for {@code java.lang.Boolean} instances that represent {@code true} or {@code 1237} for
   * {@code java.lang.Boolean} instances that represent {@code false}.
   * </p>
   * <p>
   * This is in accordance with the <i>Effective Java</i> design.
   * </p>
   *
   * @param value the boolean to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final boolean value) {
    iTotal = iTotal * iConstant + (value ? 0 : 1);
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code boolean} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final boolean[] array) {
    for (final boolean element : array) {
      append(element);
    }
    return this;
  }

  // -------------------------------------------------------------------------

  /**
   * <p>
   * Append a {@code hashCode} for a {@code byte}.
   * </p>
   *
   * @param value the byte to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final byte value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  // -------------------------------------------------------------------------

  /**
   * <p>
   * Append a {@code hashCode} for a {@code byte} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final byte[] array) {
    for (final byte element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code char}.
   * </p>
   *
   * @param value the char to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final char value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code char} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final char[] array) {
    for (final char element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code double}.
   * </p>
   *
   * @param value the double to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final double value) {
    return append(doubleToLongBits(value));
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code double} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final double[] array) {
    for (final double element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code float}.
   * </p>
   *
   * @param value the float to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final float value) {
    iTotal = iTotal * iConstant + Float.floatToIntBits(value);
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code float} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final float[] array) {
    for (final float element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for an {@code int}.
   * </p>
   *
   * @param value the int to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final int value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for an {@code int} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final int[] array) {
    for (final int element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code long}.
   * </p>
   *
   * @param value the long to add to the {@code hashCode}
   * @return this
   */
  // NOTE: This method uses >> and not >>> as Effective Java and
  // Long.hashCode do. Ideally we should switch to >>> at
  // some stage. There are backwards compat issues, so
  // that will have to wait for the time being. cf LANG-342.
  public EfficientHashCode append(final long value) {
    iTotal = iTotal * iConstant + ((int) (value ^ (value >> 32)));
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code long} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final long[] array) {
    for (final long element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for an {@code Object}.
   * </p>
   *
   * @param object the Object to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final Object object) {
    if (object == null) {
      iTotal = iTotal * iConstant;

    } else {
      if (object.getClass().isArray()) {
        appendArray(object);
      } else {
        iTotal = iTotal * iConstant + object.hashCode();
      }
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for an array.
   * </p>
   *
   * @param object the array to add to the {@code hashCode}
   */
  private void appendArray(final Object object) {
    // 'Switch' on type of array, to dispatch to the correct handler
    // This handles multi dimensional arrays
    if (object instanceof long[]) {
      append((long[]) object);
    } else if (object instanceof int[]) {
      append((int[]) object);
    } else if (object instanceof short[]) {
      append((short[]) object);
    } else if (object instanceof char[]) {
      append((char[]) object);
    } else if (object instanceof byte[]) {
      append((byte[]) object);
    } else if (object instanceof double[]) {
      append((double[]) object);
    } else if (object instanceof float[]) {
      append((float[]) object);
    } else if (object instanceof boolean[]) {
      append((boolean[]) object);
    } else {
      // Not an array of primitives
      append((Object[]) object);
    }
  }

  /**
   * <p>
   * Append a {@code hashCode} for an {@code Object} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final Object[] array) {
    for (final Object element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code short}.
   * </p>
   *
   * @param value the short to add to the {@code hashCode}
   * @return this
   */
  public EfficientHashCode append(final short value) {
    iTotal = iTotal * iConstant + value;
    return this;
  }

  /**
   * <p>
   * Append a {@code hashCode} for a {@code short} array.
   * </p>
   *
   * @param array the array to add to the {@code hashCode}
   * @return this
   */
  private EfficientHashCode append(final short[] array) {
    for (final short element : array) {
      append(element);
    }
    return this;
  }

  /**
   * <p>
   * Returns the computed {@code hashCode}.
   * </p>
   *
   * @return {@code hashCode} based on the fields appended
   */
  @Override
  public Integer build() {
    return iTotal;
  }
}
