/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.internal.builder;


import static java.lang.String.format;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.catalog.internal.DefaultTypeResolver;
import org.mule.metadata.catalog.internal.model.loaders.TypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.example.JsonTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.example.XmlTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.CsvTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.JavaTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.JsonRamlTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.JsonSchemaTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.MapTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.XmlRamlTypeLoaderFactory;
import org.mule.metadata.catalog.internal.model.loaders.shape.XmlSchemaTypeLoaderFactory;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

public class TypesCatalogBuilder {

  /**
   * {@link TypeLoaderFactory}s that are schema/shape based
   */
  private final TypeLoaderRegistry shapesRegistry;
  /**
   * {@link TypeLoaderFactory}s that are example based
   */
  private final TypeLoaderRegistry examplesRegistry;

  private final List<TypesResolverBuilder> typesResolverBuilders;
  private URI baseUri;

  public TypesCatalogBuilder(URI baseUri, ClassLoader classLoader) {
    this.baseUri = baseUri;

    this.shapesRegistry = new TypeLoaderRegistry(Arrays.asList(new JsonSchemaTypeLoaderFactory(),
                                                               new XmlSchemaTypeLoaderFactory(),
                                                               new JavaTypeLoaderFactory(classLoader),
                                                               new CsvTypeLoaderFactory(),
                                                               new MapTypeLoaderFactory(),
                                                               new JsonRamlTypeLoaderFactory(),
                                                               new XmlRamlTypeLoaderFactory()));

    this.examplesRegistry = new TypeLoaderRegistry(Arrays.asList(new JsonTypeLoaderFactory(),
                                                                 new XmlTypeLoaderFactory()));
    typesResolverBuilders = new ArrayList<>();
  }

  public void addTypesResolver(Consumer<TypesResolverBuilder> typesResolverBuilderConsumer) {
    TypesResolverBuilder typesResolverBuilder = new TypesResolverBuilder(baseUri);
    typesResolverBuilderConsumer.accept(typesResolverBuilder);
    typesResolverBuilders.add(typesResolverBuilder);
  }

  public DefaultTypeResolver build() {
    final Map<String, TypeLoader> loaders = new HashMap<>();
    for (TypesResolverBuilder typesResolverBuilder : typesResolverBuilders) {
      try {
        loaders.put(typesResolverBuilder.getCatalogName(), typesResolverBuilder.build(shapesRegistry, examplesRegistry));
      } catch (Exception e) {
        throw new RuntimeException(format("Failed to add type declaration '%s' to types catalog. Original error was: %s.",
                                          typesResolverBuilder.getCatalogName(), e.getMessage()),
                                   e);
      }
    }
    return new DefaultTypeResolver(loaders);
  }
}
