/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.internal.loader;

import org.mule.metadata.catalog.internal.DefaultTypeResolver;
import org.mule.metadata.catalog.internal.builder.TypesCatalogBuilder;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Optional;

public class TypeResolverLoader {

  private static final String NS_TYPES = "http://www.mulesoft.org/schema/mule/types";
  private static final QName ELEM_MULE = new QName(NS_TYPES, "catalogs");
  private static final QName ELEM_CATALOG = new QName(NS_TYPES, "catalog");
  private static final String ELEM_CATALOG_ATTR_NAME = "name";
  private static final String ELEM_CATALOG_ATTR_FORMAT = "format";

  private static final QName ELEM_SCHEMA = new QName(NS_TYPES, "schema");
  private static final String ELEM_SCHEMA_ATTR_FORMAT = "format";
  private static final String ELEM_SCHEMA_ATTR_LOCATION = "location";

  private static final QName ELEM_EXAMPLE = new QName(NS_TYPES, "example");
  private static final String ELEM_EXAMPLE_ATTR_FORMAT = "format";
  private static final String ELEM_EXAMPLE_ATTR_LOCATION = "location";

  public DefaultTypeResolver load(ClassLoader classLoader) {
    TypesCatalogBuilder typesCatalogBuilder = new TypesCatalogBuilder(null, classLoader);
    return typesCatalogBuilder.build();
  }

  public DefaultTypeResolver load(URL url, ClassLoader classLoader) {
    URI baseUri = null;

    try {
      baseUri = url.toURI();
    } catch (URISyntaxException e) {
      // do nothing
    }


    TypesCatalogBuilder typesCatalogBuilder = getTypesCatalogBuilder(classLoader, baseUri);
    load(url, typesCatalogBuilder);
    return typesCatalogBuilder.build();
  }

  protected TypesCatalogBuilder getTypesCatalogBuilder(ClassLoader classLoader, URI baseUri) {
    return new TypesCatalogBuilder(baseUri, classLoader);
  }

  public DefaultTypeResolver load(String data, ClassLoader classLoader) {
    TypesCatalogBuilder typesCatalogBuilder = new TypesCatalogBuilder(null, classLoader);
    load(data, typesCatalogBuilder);
    return typesCatalogBuilder.build();
  }

  private void load(URL url, TypesCatalogBuilder typesCatalogBuilder) {
    try {
      try (InputStream inputStream = url.openStream()) {
        final Element documentElement = parseRootElement(new InputSource(inputStream));
        load(documentElement, typesCatalogBuilder);
      }
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private void load(String data, TypesCatalogBuilder typesCatalogBuilder) {
    try {
      load(parseRootElement(new InputSource(new StringReader(data))), typesCatalogBuilder);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private void load(Element documentElement, TypesCatalogBuilder typesCatalogBuilder) {
    XmlMatcher.match(documentElement, ELEM_MULE).ifPresent(xmlMatcher -> {
      xmlMatcher.matchMany(ELEM_CATALOG).forEach(catalog -> {
        catalog.matchMany(ELEM_SCHEMA).forEach(schema -> {
          typesCatalogBuilder.addTypesResolver(typesResolverBuilder -> {
            catalog.matchAttribute(ELEM_CATALOG_ATTR_NAME).ifPresent(typesResolverBuilder::catalogName);
            catalog.matchAttribute(ELEM_CATALOG_ATTR_FORMAT).ifPresent(typesResolverBuilder::catalogFormat);
            schema.matchAttribute(ELEM_SCHEMA_ATTR_FORMAT).ifPresent(typesResolverBuilder::schemaFormat);
            final Optional<String> locationAttribute = schema.matchAttribute(ELEM_SCHEMA_ATTR_LOCATION);
            if (locationAttribute.isPresent()) {
              locationAttribute.ifPresent(typesResolverBuilder::schemaLocation);
            } else {
              typesResolverBuilder.schemaContent(schema.value());
            }
          });
        });

        catalog.matchMany(ELEM_EXAMPLE).forEach(example -> {
          typesCatalogBuilder.addTypesResolver(typesResolverBuilder -> {
            catalog.matchAttribute(ELEM_CATALOG_ATTR_NAME).ifPresent(typesResolverBuilder::catalogName);
            catalog.matchAttribute(ELEM_CATALOG_ATTR_FORMAT).ifPresent(typesResolverBuilder::catalogFormat);
            example.matchAttribute(ELEM_EXAMPLE_ATTR_FORMAT).ifPresent(typesResolverBuilder::exampleFormat);
            final Optional<String> locationAttribute = example.matchAttribute(ELEM_EXAMPLE_ATTR_LOCATION);
            if (locationAttribute.isPresent()) {
              locationAttribute.ifPresent(typesResolverBuilder::exampleLocation);
            } else {
              typesResolverBuilder.exampleContent(example.value());
            }
          });
        });
      });
    });
  }

  private Element parseRootElement(InputSource inputSource) throws ParserConfigurationException, SAXException, IOException {
    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
    documentBuilderFactory.setNamespaceAware(true);
    documentBuilderFactory.setValidating(false);
    DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
    Document document = documentBuilder.parse(inputSource);
    return document.getDocumentElement();
  }
}
