/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.api;

import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.metadata.api.utils.MetadataTypeUtils.checkArgument;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.ast.internal.DefaultObjectFieldHandler;
import org.mule.metadata.ast.internal.TypeMirrorLoaderVisitor;
import org.mule.metadata.ast.internal.handler.ArrayTypeHandler;
import org.mule.metadata.ast.internal.handler.BinaryTypeHandler;
import org.mule.metadata.ast.internal.handler.BooleanTypeHandler;
import org.mule.metadata.ast.internal.handler.DateTypeHandler;
import org.mule.metadata.ast.internal.handler.EnumTypeHandler;
import org.mule.metadata.ast.internal.handler.MapTypeHandler;
import org.mule.metadata.ast.internal.handler.NumberTypeHandler;
import org.mule.metadata.ast.internal.handler.StringTypeHandler;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.type.TypeMirror;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * {@link TypeLoader} implementation which works with the Java AST.
 *
 * @since 1.1.0
 */
public final class ASTTypeLoader implements TypeLoader {

  private ProcessingEnvironment processingEnvironment;
  private final ObjectFieldHandler objectFieldHandler;
  private List<TypeHandler> handlers;

  public ASTTypeLoader(ProcessingEnvironment processingEnvironment) {
    this(processingEnvironment, emptyList(), new DefaultObjectFieldHandler(processingEnvironment));
  }

  public ASTTypeLoader(ProcessingEnvironment processingEnvironment, List<TypeHandler> handlers,
                       ObjectFieldHandler objectFieldHandler) {
    checkArgument(handlers != null, "Handlers can't be null");
    checkArgument(objectFieldHandler != null, "ObjectFieldHandler can't be null");
    this.processingEnvironment = processingEnvironment;
    this.objectFieldHandler = objectFieldHandler;

    this.handlers = new ArrayList<>();
    this.handlers.add(new StringTypeHandler(processingEnvironment));
    this.handlers.add(new BinaryTypeHandler(processingEnvironment));
    this.handlers.add(new NumberTypeHandler(processingEnvironment));
    this.handlers.add(new ArrayTypeHandler(processingEnvironment));
    this.handlers.add(new BooleanTypeHandler(processingEnvironment));
    this.handlers.add(new DateTypeHandler(processingEnvironment));
    this.handlers.add(new MapTypeHandler(processingEnvironment));
    this.handlers.add(new EnumTypeHandler(processingEnvironment));
    this.handlers.addAll(handlers);
  }

  /**
   * Given a {@link TypeMirror} introspect it an returns the correspondent {@link MetadataType}
   *
   * @param typeMirror type to introspect
   * @return an optional {@link MetadataType}
   */
  public Optional<MetadataType> load(TypeMirror typeMirror) {
    TypeBuilder<?> accept =
        typeMirror.accept(new TypeMirrorLoaderVisitor(processingEnvironment, handlers, objectFieldHandler),
                          new IntrospectionContext());
    return accept != null ? of(accept.build()) : empty();
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    throw new UnsupportedOperationException("This method is not supported, please use #load(TypeMirror)");
  }
}
