/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal;

import static org.mule.metadata.api.annotation.Accessibility.READ_ONLY;
import static org.mule.metadata.api.annotation.Accessibility.WRITE_ONLY;
import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.ast.api.IntrospectionContext;
import org.mule.metadata.ast.api.ObjectFieldHandler;
import org.mule.metadata.ast.api.Property;
import org.mule.metadata.ast.api.TypeUtils;

import java.util.Comparator;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.type.TypeVisitor;

/**
 * Default {@link ObjectFieldHandler} implementation for {@link ObjectType} fields
 *
 * @since 4.1
 */
public final class DefaultObjectFieldHandler implements ObjectFieldHandler {

  private ProcessingEnvironment processingEnvironment;

  public DefaultObjectFieldHandler(ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
  }

  @Override
  public void handle(Element element, ObjectTypeBuilder objectTypeBuilder, IntrospectionContext context,
                     TypeVisitor<TypeBuilder<?>, IntrospectionContext> typeVisitor) {

    TypeUtils.getProperties(element, processingEnvironment)
        .stream()
        .sorted(Comparator.comparing(Property::getBeanName))
        .forEach(property -> {
          ObjectFieldTypeBuilder fieldTypeBuilder = objectTypeBuilder.addField()
              .key(property.getBeanName());

          Element propertyElement = processingEnvironment.getTypeUtils().asElement(property.getType());
          if (context.contains(propertyElement)) {
            fieldTypeBuilder.value(context.get(propertyElement));
          } else {
            fieldTypeBuilder.value(property.getType().accept(typeVisitor, context));
          }
          addAccessibilityProperty(fieldTypeBuilder, property.getGetterMethod() != null, property.getSetterMethod() != null);
        });
  }

  private void addAccessibilityProperty(ObjectFieldTypeBuilder fieldTypeBuilder, boolean hasGetter, boolean hasSetter) {
    if (hasGetter && !hasSetter) {
      fieldTypeBuilder.with(new AccessibilityAnnotation(READ_ONLY));
    }
    if (!hasGetter && hasSetter) {
      fieldTypeBuilder.with(new AccessibilityAnnotation(WRITE_ONLY));
    }
  }
}
