/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal;

import static javax.lang.model.type.TypeKind.BOOLEAN;
import static javax.lang.model.type.TypeKind.BYTE;
import static javax.lang.model.type.TypeKind.CHAR;
import static javax.lang.model.type.TypeKind.DOUBLE;
import static javax.lang.model.type.TypeKind.FLOAT;
import static javax.lang.model.type.TypeKind.INT;
import static javax.lang.model.type.TypeKind.LONG;
import static javax.lang.model.type.TypeKind.SHORT;
import static javax.lang.model.type.TypeKind.VOID;
import static org.apache.commons.lang3.SystemUtils.IS_JAVA_1_8;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.type.ArrayType;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Elements;
import javax.lang.model.util.Types;

/**
 * Utility class for AST Stuff
 *
 * @since 1.1.0
 */
public class ASTHelper {

  private ProcessingEnvironment processingEnvironment;
  private Map<String, TypeMirror> primitiveTypeMirrors = new HashMap<>();
  private MuleLazyValue<ASTHelperJDK9> astHelperJDK9;

  public ASTHelper(ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;

    primitiveTypeMirrors.put("int", processingEnvironment.getTypeUtils().getPrimitiveType(INT));
    primitiveTypeMirrors.put("long", processingEnvironment.getTypeUtils().getPrimitiveType(LONG));
    primitiveTypeMirrors.put("double", processingEnvironment.getTypeUtils().getPrimitiveType(DOUBLE));
    primitiveTypeMirrors.put("float", processingEnvironment.getTypeUtils().getPrimitiveType(FLOAT));
    primitiveTypeMirrors.put("boolean", processingEnvironment.getTypeUtils().getPrimitiveType(BOOLEAN));
    primitiveTypeMirrors.put("char", processingEnvironment.getTypeUtils().getPrimitiveType(CHAR));
    primitiveTypeMirrors.put("byte", processingEnvironment.getTypeUtils().getPrimitiveType(BYTE));
    primitiveTypeMirrors.put("void", processingEnvironment.getTypeUtils().getNoType(VOID));
    primitiveTypeMirrors.put("short", processingEnvironment.getTypeUtils().getPrimitiveType(SHORT));

    astHelperJDK9 = new MuleLazyValue<>(() -> new ASTHelperJDK9(processingEnvironment));
  }

  public boolean isAssignable(ArrayType arrayType, Class... classes) {
    for (Class aClass : classes) {
      if (aClass.isArray() && isAssignable(arrayType.getComponentType(), aClass.getComponentType())) {
        return true;
      }
    }

    return false;
  }

  public boolean isAssignable(TypeMirror typeMirror, Class... classes) {
    Types utils = processingEnvironment.getTypeUtils();

    for (Class aClass : classes) {
      if (utils.isAssignable(utils.erasure(typeMirror),
                             utils.erasure(getTypeMirror(aClass)))) {
        return true;
      }
    }

    return false;
  }

  private TypeMirror getTypeMirror(Class aClass) {
    if (aClass.isPrimitive()) {
      return primitiveTypeMirrors.get(aClass.getCanonicalName());
    } else {
      if (shouldUseJavaModules()) {
        return astHelperJDK9.get().getTypeElement(aClass).asType();
      } else {
        Elements elementUtils = processingEnvironment.getElementUtils();
        return elementUtils.getTypeElement(aClass.getName()).asType();
      }
    }
  }

  public static String typeId(DeclaredType type) {
    return type.asElement().toString();
  }

  private boolean shouldUseJavaModules() {
    return !IS_JAVA_1_8;
  }
}
