/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.metadata.ast.internal.ASTHelper.typeId;
import static org.mule.metadata.ast.internal.ClassInformationAnnotationFactory.fromTypeMirror;
import static org.mule.metadata.java.internal.handler.StringHandler.CHAR_LENGTH;

import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.ast.api.IntrospectionContext;
import org.mule.metadata.ast.api.ObjectFieldHandler;
import org.mule.metadata.ast.api.TypeHandler;

import java.util.List;
import java.util.Optional;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.type.ArrayType;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.ErrorType;
import javax.lang.model.type.ExecutableType;
import javax.lang.model.type.IntersectionType;
import javax.lang.model.type.NoType;
import javax.lang.model.type.NullType;
import javax.lang.model.type.PrimitiveType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.TypeVariable;
import javax.lang.model.type.TypeVisitor;
import javax.lang.model.type.UnionType;
import javax.lang.model.type.WildcardType;
import javax.lang.model.util.Types;

/**
 * {@link TypeVisitor} which navigates throught a {@link TypeMirror} and returns the correspondent {@link TypeBuilder}
 *
 * @since 1.1.0
 */
public final class TypeMirrorLoaderVisitor implements TypeVisitor<TypeBuilder<?>, IntrospectionContext> {

  private final ASTHelper astHelper;
  private final ProcessingEnvironment processingEnvironment;
  private final List<TypeHandler> handlers;
  private final TypeMirror objectType;
  private final ObjectFieldHandler objectFieldHandler;
  private final Types typeUtils;

  public TypeMirrorLoaderVisitor(ProcessingEnvironment processingEnvironment, List<TypeHandler> handlers,
                                 ObjectFieldHandler objectFieldHandler) {
    this.processingEnvironment = processingEnvironment;
    this.handlers = handlers;
    this.objectType = processingEnvironment.getElementUtils().getTypeElement(Object.class.getCanonicalName()).asType();
    this.astHelper = new ASTHelper(processingEnvironment);
    this.objectFieldHandler = objectFieldHandler;
    this.typeUtils = processingEnvironment.getTypeUtils();
  }

  @Override
  public TypeBuilder<?> visit(TypeMirror t, IntrospectionContext s) {
    return null;
  }

  @Override
  public TypeBuilder<?> visit(TypeMirror t) {
    return null;
  }

  @Override
  public TypeBuilder<?> visitPrimitive(PrimitiveType t, IntrospectionContext s) {
    switch (t.getKind()) {
      case LONG:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment)).integer();
      case INT:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment)).integer();
      case SHORT:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment));
      case DOUBLE:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment));
      case FLOAT:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment));
      case CHAR:
        return builder().stringType().with(fromTypeMirror(t, processingEnvironment))
            .with(new LengthAnnotation(CHAR_LENGTH, CHAR_LENGTH));
      case VOID:
        return builder().voidType();
      case NULL:
        return builder().nullType();
      case BYTE:
        return builder().numberType().with(fromTypeMirror(t, processingEnvironment));
      case BOOLEAN:
        return builder().booleanType().id(Boolean.TYPE.getTypeName());
    }

    throw new IllegalArgumentException("Unknown Primitive Type " + t);
  }

  @Override
  public TypeBuilder<?> visitNull(NullType t, IntrospectionContext s) {
    return builder().nullType();
  }

  @Override
  public TypeBuilder<?> visitArray(ArrayType t, IntrospectionContext s) {
    if (astHelper.isAssignable(t, Byte[].class, byte[].class)) {
      return builder().binaryType().with(fromTypeMirror(t, processingEnvironment));
    }

    return builder().arrayType().of(t.getComponentType().accept(this, s)).with(fromTypeMirror(t, processingEnvironment));
  }

  @Override
  public TypeBuilder<?> visitDeclared(DeclaredType t, IntrospectionContext context) {
    for (TypeHandler handler : handlers) {
      if (handler.handles(t)) {
        return handler.handle(t, this, context);
      }
    }

    Element element = t.asElement();
    if (context.contains(element)) {
      return context.get(element);
    }

    ObjectTypeBuilder objectTypeBuilder = builder().objectType().id(typeId(t)).with(fromTypeMirror(t, processingEnvironment));

    extractJavadoc(processingEnvironment, element)
        .ifPresent(objectTypeBuilder::description);

    context.push(element, objectTypeBuilder);

    if (!typeUtils.isSameType(t, objectType)) {
      objectFieldHandler.handle(element, objectTypeBuilder, context, this);
    }

    context.pop();

    return objectTypeBuilder;
  }

  private Optional<String> extractJavadoc(ProcessingEnvironment processingEnv, Element element) {
    String comment = processingEnv.getElementUtils().getDocComment(element);
    if (comment == null || comment.trim().isEmpty()) {
      return Optional.empty();
    }

    return Optional.of(comment.trim().replaceAll("\\{@[^ ]+ ([^\\}]+)\\}", "$1"));
  }

  private BaseTypeBuilder builder() {
    return BaseTypeBuilder.create(JAVA);
  }

  @Override
  public TypeBuilder<?> visitError(ErrorType t, IntrospectionContext s) {
    return null;
  }

  @Override
  public TypeBuilder<?> visitTypeVariable(TypeVariable t, IntrospectionContext s) {
    if (!t.getLowerBound().equals(typeUtils.getNullType())) {
      return t.getLowerBound().accept(this, s);
    } else if (!t.getUpperBound().equals(typeUtils.getNullType())) {
      return t.getUpperBound().accept(this, s);
    } else {
      return objectType.accept(this, s);
    }
  }

  @Override
  public TypeBuilder<?> visitWildcard(WildcardType t, IntrospectionContext s) {
    if (t.getExtendsBound() != null) {
      return t.getExtendsBound().accept(this, s);
    } else if (t.getSuperBound() != null) {
      return t.getSuperBound().accept(this, s);
    } else {
      return objectType.accept(this, s);
    }
  }

  @Override
  public TypeBuilder<?> visitExecutable(ExecutableType t, IntrospectionContext s) {
    return null;
  }

  @Override
  public TypeBuilder<?> visitNoType(NoType t, IntrospectionContext s) {
    return builder().voidType();
  }

  @Override
  public TypeBuilder<?> visitUnknown(TypeMirror t, IntrospectionContext s) {
    return builder().voidType();
  }

  @Override
  public TypeBuilder<?> visitUnion(UnionType t, IntrospectionContext s) {
    return builder().anyType();
  }

  @Override
  public TypeBuilder<?> visitIntersection(IntersectionType t, IntrospectionContext s) {
    return builder().anyType();
  }
}
