/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal.handler;

import static org.mule.metadata.api.utils.MetadataTypeUtils.checkArgument;
import static org.mule.metadata.ast.internal.ASTHelper.typeId;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.ast.api.TypeHandler;
import org.mule.metadata.ast.api.IntrospectionContext;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.TypeVisitor;
import javax.xml.datatype.XMLGregorianCalendar;

import java.time.chrono.ChronoLocalDate;
import java.time.temporal.Temporal;
import java.util.Calendar;
import java.util.Date;

/**
 * {@link TypeHandler} for {@link DateTimeType} and {@link DateType}
 *
 * @since 1.1.0
 */
public class DateTypeHandler extends BaseTypeHandler {

  public DateTypeHandler(ProcessingEnvironment processingEnvironment) {
    super(processingEnvironment, Calendar.class, XMLGregorianCalendar.class, Temporal.class, Date.class, java.sql.Date.class);
  }

  @Override
  public TypeBuilder<?> handle(TypeMirror typeMirror, TypeVisitor<TypeBuilder<?>, IntrospectionContext> visitor,
                               IntrospectionContext context) {
    checkArgument(typeMirror instanceof DeclaredType, "The given type should be a DeclaredType");
    DeclaredType declaredType = (DeclaredType) typeMirror;
    if (astHelper.isAssignable(declaredType, Date.class, java.sql.Date.class, ChronoLocalDate.class)) {
      return builder().dateType().id(typeId(declaredType));
    }

    if (astHelper.isAssignable(declaredType, Calendar.class, XMLGregorianCalendar.class, Temporal.class)) {
      return builder().dateTimeType()
          .id(typeId(declaredType));
    }
    throw new IllegalArgumentException("The given TypeMirror is not handleable. " + typeMirror);
  }
}
