/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;
import javax.lang.model.util.Elements;

/**
 * AST Helper that works with JDK 9 and Upwards but being compile compatible with previous JDK versions.
 *
 * @since 1.2.0
 */
class ASTHelperJDK9 {

  private Method getModuleElementMethod;
  private Method getTypeMethodElementMethod;
  private Method getModuleMethod;
  private Method getModuleNameMethod;
  private ProcessingEnvironment processingEnvironment;

  ASTHelperJDK9(ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
    try {
      getModuleElementMethod = Elements.class.getMethod("getModuleElement", CharSequence.class);
      getTypeMethodElementMethod =
          Elements.class.getMethod("getTypeElement", Class.forName("javax.lang.model.element.ModuleElement"), CharSequence.class);
      getModuleMethod = Class.class.getMethod("getModule");
      getModuleNameMethod = Class.forName("java.lang.Module").getMethod("getName");
    } catch (Exception e) {
      throw new IllegalStateException("An unexpected error occurred trying to initialize this class, this is probably you are using a JDK 8 or previous.",
                                      e);
    }
  }

  TypeElement getTypeElement(Class aClass) {
    try {
      Elements elementUtils = processingEnvironment.getElementUtils();

      Object module = getModuleMethod.invoke(aClass);
      Object moduleName = getModuleNameMethod.invoke(module);
      Object moduleElement = getModuleElementMethod.invoke(elementUtils, moduleName);
      TypeElement typeElement = (TypeElement) getTypeMethodElementMethod.invoke(elementUtils, moduleElement, aClass.getName());

      return typeElement;

    } catch (IllegalAccessException | InvocationTargetException e) {
      throw new IllegalStateException("Unexpected error occurred trying to Create TypeElement for class: " + aClass, e);
    }
  }
}
