/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.utils;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.metadata.api.utils.MetadataTypeUtils;

import java.util.Optional;

import org.apache.commons.lang3.ClassUtils;

public final class JavaTypeUtils {

  private JavaTypeUtils() {}

  public static <T> Class<T> getType(MetadataType metadataType) {
    return getType(metadataType, Thread.currentThread().getContextClassLoader());
  }

  public static <T> Class<T> getType(MetadataType metadataType, ClassLoader classLoader) {
    if (metadataType instanceof NullType || metadataType instanceof VoidType) {
      return (Class<T>) void.class;
    }

    Optional<String> typeId = MetadataTypeUtils.getTypeId(metadataType);
    if (!typeId.isPresent()) {
      throw new IllegalArgumentException("metadataType does not contain a type annotation");
    }

    try {
      return (Class<T>) ClassUtils.getClass(classLoader, typeId.get(), true);
    } catch (ClassNotFoundException e) {
      throw new IllegalArgumentException("Could not load class " + typeId.get(), e);
    }
  }

  public static Optional<MetadataType> getGenericTypeAt(MetadataType metadataType, int index, ClassTypeLoader typeLoader) {
    Optional<ClassInformationAnnotation> classInformationAnnotation =
        metadataType.getAnnotation(ClassInformationAnnotation.class);
    if (!classInformationAnnotation.isPresent() || classInformationAnnotation.get().getGenericTypes().size() <= index) {
      return Optional.empty();
    }
    return typeLoader.load(classInformationAnnotation.get().getGenericTypes().get(index));
  }
}
