/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.annotation;

import static java.util.Arrays.stream;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.builder.EqualsBuilder.reflectionEquals;
import static org.apache.commons.lang3.builder.HashCodeBuilder.reflectionHashCode;
import static org.apache.commons.lang3.builder.ToStringBuilder.reflectionToString;
import static org.apache.commons.lang3.builder.ToStringStyle.SIMPLE_STYLE;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.java.api.utils.TypeResolver;

import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Optional;

public class ClassInformationAnnotation implements TypeAnnotation {

  public static final String NAME = "classInformation";

  private final boolean hasDefaultConstructor;
  private final boolean isInterface;
  private final boolean isInstantiable;
  private final boolean isAbstract;
  private final boolean isFinal;
  private final List<String> implementedInterfaces;
  private final String parent;
  private final List<String> genericTypes;

  public ClassInformationAnnotation(Class<?> clazz, List<Type> genericTypes) {
    this.implementedInterfaces = getImplementedInterfaces(clazz);
    this.parent = getParentClass(clazz);
    this.isFinal = Modifier.isFinal(clazz.getModifiers());
    this.isAbstract = Modifier.isAbstract(clazz.getModifiers());
    this.isInterface = clazz.isInterface();
    this.hasDefaultConstructor = hasDefaultConstructor(clazz);
    this.genericTypes = getGenerics(genericTypes);
    this.isInstantiable = !isInterface && !isAbstract && hasDefaultConstructor;
  }

  private List<String> getGenerics(List<Type> genericTypes) {
    if (genericTypes != null && !genericTypes.isEmpty()) {
      return unmodifiableList(genericTypes.stream()
          .map(TypeResolver::erase)
          .map(Type::getTypeName)
          .collect(toList()));
    }
    return emptyList();
  }

  private List<String> getImplementedInterfaces(Class<?> clazz) {
    return unmodifiableList(stream(clazz.getInterfaces())
        .map(Class::getCanonicalName)
        .filter(name -> name != null)
        .collect(toList()));
  }

  private String getParentClass(Class<?> clazz) {
    Class<?> parent = clazz.getSuperclass();
    return parent != null && !parent.equals(Object.class) ? parent.getName() : "";
  }

  private boolean hasDefaultConstructor(Class<?> clazz) {
    return !isInterface && stream(clazz.getDeclaredConstructors())
        .anyMatch(c -> c.getParameterCount() == 0
            && Modifier.isPublic(c.getModifiers()));

  }

  @Override
  public String getName() {
    return NAME;
  }

  public boolean isInstantiable() {
    return isInstantiable;
  }

  public List<String> getGenericTypes() {
    return genericTypes;
  }

  public Optional<String> getParent() {
    return isNotBlank(parent) ? of(parent) : empty();
  }

  public boolean isInterface() {
    return isInterface;
  }

  public boolean isFinal() {
    return isFinal;
  }

  public boolean isAbstract() {
    return isAbstract;
  }

  public List<String> getImplementedInterfaces() {
    return implementedInterfaces;
  }

  public boolean hasDefaultConstructor() {
    return hasDefaultConstructor;
  }

  @Override
  public boolean equals(Object obj) {
    return reflectionEquals(this, obj);
  }

  @Override
  public int hashCode() {
    return reflectionHashCode(this);
  }

  @Override
  public String toString() {
    return reflectionToString(this, SIMPLE_STYLE);
  }
}
