/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.internal.handler;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static org.mule.metadata.java.api.utils.TypeResolver.erase;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.metadata.java.api.handler.ClassHandler;
import org.mule.metadata.java.api.handler.TypeHandlerManager;
import org.mule.metadata.java.api.utils.ParsingContext;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

public class MapClassHandler implements ClassHandler {

  @Override
  public boolean handles(Class<?> clazz) {
    return Map.class.isAssignableFrom(clazz);
  }

  @Override
  public TypeBuilder<?> handleClass(Class<?> clazz,
                                    List<Type> genericTypes, TypeHandlerManager typeHandlerManager,
                                    ParsingContext context, BaseTypeBuilder typeBuilder) {

    final ObjectTypeBuilder objectTypeBuilder = typeBuilder.objectType().id(clazz.getName());
    if (genericTypes.isEmpty()) {
      //If no generic look in the inherited interfaces
      implementedInterfacesGenericsOrDefault(clazz, typeHandlerManager, context, objectTypeBuilder);
    } else if (genericTypes.size() != 2) {
      throw new IllegalArgumentException(String.format("Exactly 2 generics were expected but %d were provided instead",
                                                       genericTypes.size()));
    } else {
      openWith(typeHandlerManager, context, objectTypeBuilder, genericTypes.get(1));
    }
    objectTypeBuilder.with(new ClassInformationAnnotation(clazz, genericTypes));
    return objectTypeBuilder;
  }

  private void implementedInterfacesGenericsOrDefault(Class<?> clazz, TypeHandlerManager typeHandlerManager,
                                                      ParsingContext context,
                                                      ObjectTypeBuilder objectTypeBuilder) {
    final List<Type> superclassGenerics = getImplementedInterfaceGenerics(clazz);
    if (superclassGenerics.size() == 2) {
      openWith(typeHandlerManager, context, objectTypeBuilder, superclassGenerics.get(1));
    } else {
      objectTypeBuilder.open();
    }
  }

  private void openWith(TypeHandlerManager typeHandlerManager, ParsingContext context, ObjectTypeBuilder objectTypeBuilder,
                        Type type) {
    final Optional<TypeBuilder<?>> keyTypeBuilder = context.getTypeBuilder(type);
    if (keyTypeBuilder.isPresent()) {
      objectTypeBuilder.openWith(keyTypeBuilder.get());
    } else {
      typeHandlerManager.handle(type, context, objectTypeBuilder.openWith());
    }
  }

  private List<Type> getImplementedInterfaceGenerics(Class clazz) {
    return getImplementedInterfaceGenerics(clazz, new HashSet<>());
  }

  private List<Type> getImplementedInterfaceGenerics(Class clazz, Set<Class> analyzedClasses) {
    while (clazz != null && !Object.class.equals(clazz)) {
      for (Type interfaceType : clazz.getGenericInterfaces()) {
        Class interfaceClazz = erase(interfaceType);
        if (analyzedClasses.contains(interfaceClazz)) {
          continue;
        } else {
          analyzedClasses.add(interfaceClazz);
        }
        if (interfaceClazz.equals(Map.class)) {
          final ParameterizedType paramType = (ParameterizedType) interfaceType;
          return asList(paramType.getActualTypeArguments());
        } else {
          List<Type> interfaceGenerics = getImplementedInterfaceGenerics(interfaceClazz, analyzedClasses);
          if (!interfaceGenerics.isEmpty()) {
            return interfaceGenerics;
          }
        }
      }

      clazz = clazz.getSuperclass();
    }

    return emptyList();
  }
}
