/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.utils;

import static java.util.Optional.ofNullable;
import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.IntAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.utils.MetadataTypeUtils;
import org.mule.metadata.api.utils.MetadataTypeUtils.TypeResolverVisitor;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.ClassUtils;

public final class JavaTypeUtils {

  private JavaTypeUtils() {}

  public static Optional<String> getId(MetadataType metadataType) {
    Optional<String> id = MetadataTypeUtils.getTypeId(metadataType);
    if (id.isPresent()) {
      return id;
    }

    TypeResolverVisitor visitor = new TypeResolverVisitor() {

      @Override
      public void defaultVisit(MetadataType metadataType) {
        typeId = fromClassInfo(metadataType);
      }

      @Override
      public void visitAnyType(AnyType anyType) {
        typeId = Object.class.getName();
      }

      @Override
      public void visitBoolean(BooleanType booleanType) {
        typeId = fromClassInfo(metadataType);

        if (typeId == null) {
          typeId = boolean.class.getName();
        }
      }

      @Override
      public void visitNumber(NumberType numberType) {
        defaultVisit(numberType);
        if (typeId == null) {
          typeId = numberType.getAnnotation(IntAnnotation.class).map(a -> Integer.class.getName()).orElse(null);
        }
      }

      @Override
      public void visitObject(ObjectType objectType) {
        typeId = getTypeId(objectType);
        if (typeId == null) {
          defaultVisit(objectType);
        }
      }

      @Override
      public void visitString(StringType stringType) {
        if (stringType.getAnnotation(EnumAnnotation.class).isPresent()) {
          typeId = getTypeId(stringType);
        }

        if (typeId == null) {
          typeId = String.class.getName();
        }
      }

      @Override
      public void visitArrayType(ArrayType arrayType) {
        typeId = fromClassInfo(arrayType);

        if (typeId == null) {
          typeId = List.class.getName();
        }
      }

      private String fromClassInfo(MetadataType type) {
        return type.getAnnotation(ClassInformationAnnotation.class)
            .map(ClassInformationAnnotation::getClassname)
            .orElse(null);
      }

      private String getTypeId(MetadataType type) {
        return type.getAnnotation(TypeIdAnnotation.class).map(TypeIdAnnotation::getValue).orElse(null);
      }
    };

    metadataType.accept(visitor);
    return ofNullable(visitor.getResolvedTypeId());
  }

  public static <T> Class<T> getType(MetadataType metadataType) {
    return getType(metadataType, Thread.currentThread().getContextClassLoader());
  }

  public static <T> Class<T> getType(MetadataType metadataType, ClassLoader classLoader) {
    if (metadataType instanceof NullType || metadataType instanceof VoidType) {
      return (Class<T>) void.class;
    }

    return getId(metadataType).map(id -> {
      try {
        return (Class<T>) ClassUtils.getClass(classLoader, id, true);
      } catch (ClassNotFoundException e) {
        throw new IllegalArgumentException("Could not load class " + id, e);
      }
    }).orElseThrow(() -> new IllegalArgumentException("MetadataType does not contain class information"));
  }

  public static Optional<MetadataType> getGenericTypeAt(MetadataType metadataType, int index, ClassTypeLoader typeLoader) {
    Optional<ClassInformationAnnotation> classInformationAnnotation =
        metadataType.getAnnotation(ClassInformationAnnotation.class);
    if (!classInformationAnnotation.isPresent() || classInformationAnnotation.get().getGenericTypes().size() <= index) {
      return Optional.empty();
    }
    return typeLoader.load(classInformationAnnotation.get().getGenericTypes().get(index));
  }
}
