/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.metadata.java.api.utils;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.IntAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import java.util.List;
import java.util.Optional;

class ClassNameResolverVisitor extends MetadataTypeVisitor {

  private String typeName = null;

  public Optional<String> getTypeName() {
    return Optional.ofNullable(typeName);
  }

  @Override
  public void visitNull(NullType nullType) {
    typeName = void.class.getName();
  }

  @Override
  public void visitVoid(VoidType voidType) {
    typeName = void.class.getName();
  }

  @Override
  public void defaultVisit(MetadataType metadataType) {
    typeName = findFromAnnotations(metadataType).orElse(null);
  }

  @Override
  public void visitAnyType(AnyType anyType) {
    typeName = findFromAnnotations(anyType).orElse(Object.class.getName());
  }

  @Override
  public void visitBoolean(BooleanType booleanType) {
    typeName = findFromAnnotations(booleanType).orElse(boolean.class.getName());
  }

  @Override
  public void visitNumber(NumberType numberType) {
    typeName = findFromAnnotations(numberType)
        .orElse(numberType.getAnnotation(IntAnnotation.class).map(a -> Integer.class.getName()).orElse(null));
  }

  @Override
  public void visitString(StringType stringType) {
    if (stringType.getAnnotation(EnumAnnotation.class).isPresent()) {
      typeName = getTypeId(stringType).orElse(String.class.getName());
    } else if (stringType.getAnnotation(ClassInformationAnnotation.class).isPresent()) {
      typeName = stringType.getAnnotation(ClassInformationAnnotation.class).get().getClassname();
    } else {
      typeName = String.class.getName();
    }
  }

  @Override
  public void visitObjectField(ObjectFieldType objectFieldType) {
    objectFieldType.getValue().accept(this);
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    typeName = findFromAnnotations(arrayType).orElse(List.class.getName());
  }

  private Optional<String> findFromAnnotations(MetadataType type) {
    Optional<String> classInfo = type.getAnnotation(ClassInformationAnnotation.class)
        .map(ClassInformationAnnotation::getClassname);
    return classInfo.isPresent() ? classInfo : getTypeId(type);
  }
}
