/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.annotation;

import static java.util.Arrays.stream;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.java.api.utils.TypeResolver;

import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.lang.model.type.ArrayType;

public class ClassInformationAnnotation implements TypeAnnotation {

  /**
   * Classes from this packages contain information that is handled internally by the runtime and must not be exposed in the
   * extension model.
   */
  public static final String INTERNAL_METADATA_PACKAGE = "org.mule.runtime.api.component.";

  public static final String NAME = "classInformation";

  private final String classname;
  private final boolean hasDefaultConstructor;
  private final boolean isInterface;
  private final boolean isInstantiable;
  private final boolean isAbstract;
  private final boolean isFinal;
  private final List<String> implementedInterfaces;
  private final String parent;
  private final List<String> genericTypes;
  private final boolean isMap;


  public ClassInformationAnnotation(String classname, boolean hasDefaultConstructor, boolean isInterface, boolean isInstantiable,
                                    boolean isAbstract, boolean isFinal, List<String> implementedInterfaces, String parent,
                                    List<String> genericTypes, boolean isMap) {
    this.classname = classname;
    this.hasDefaultConstructor = hasDefaultConstructor;
    this.isInterface = isInterface;
    this.isInstantiable = isInstantiable;
    this.isAbstract = isAbstract;
    this.isFinal = isFinal;
    this.implementedInterfaces = implementedInterfaces;
    this.parent = parent;
    this.genericTypes = genericTypes;
    this.isMap = isMap;
  }

  public ClassInformationAnnotation(Class<?> clazz) {
    this(clazz, emptyList());
  }

  public ClassInformationAnnotation(Class<?> clazz, List<Type> genericTypes) {

    this.classname = getName(clazz);
    this.implementedInterfaces = getImplementedInterfaces(clazz);
    this.parent = getParentClass(clazz);
    this.isFinal = Modifier.isFinal(clazz.getModifiers());
    this.isAbstract = Modifier.isAbstract(clazz.getModifiers());
    this.isInterface = clazz.isInterface();
    this.hasDefaultConstructor = hasDefaultConstructor(clazz);
    this.genericTypes = getGenerics(genericTypes);
    this.isInstantiable = !isInterface && !isAbstract && hasDefaultConstructor;
    this.isMap = Map.class.isAssignableFrom(clazz);
  }

  /**
   * to correct getName()
   * @param clazz
   * @return
   */
  public static String getName(Class<?> clazz) {
    if (!(clazz.getName().startsWith("["))) {
      return clazz.getName();
    } else {
      String name = clazz.getName();
      if (!name.contains("$")) { //for arrays, to keep backwards.
        return clazz.getCanonicalName();
      } else { //for inner class arrays
        String CanonicalName = clazz.getCanonicalName();
        int indexToChange = CanonicalName.lastIndexOf(".");
        return CanonicalName.substring(0, indexToChange) + "$" + CanonicalName.substring(indexToChange + 1);
      }
    }
  }

  private List<String> getGenerics(List<Type> genericTypes) {
    if (genericTypes != null && !genericTypes.isEmpty()) {
      return unmodifiableList(genericTypes.stream()
          .map(TypeResolver::erase)
          .map(Type::getTypeName)
          .collect(toList()));
    }
    return emptyList();
  }

  private List<String> getImplementedInterfaces(Class<?> clazz) {
    return unmodifiableList(stream(clazz.getInterfaces())
        .map(Class::getCanonicalName)
        .filter(name -> name != null)
        .filter(name -> !name.startsWith(INTERNAL_METADATA_PACKAGE))
        .collect(toList()));
  }

  private String getParentClass(Class<?> clazz) {
    Class<?> parent = clazz.getSuperclass();
    return parent != null && !parent.equals(Object.class) && !parent.getName().startsWith(INTERNAL_METADATA_PACKAGE)
        ? parent.getName()
        : "";
  }

  private boolean hasDefaultConstructor(Class<?> clazz) {
    return !isInterface && stream(clazz.getDeclaredConstructors())
        .anyMatch(c -> c.getParameterCount() == 0
            && Modifier.isPublic(c.getModifiers()));

  }

  @Override
  public String getName() {
    return NAME;
  }

  public String getClassname() {
    return classname;
  }

  public boolean isInstantiable() {
    return isInstantiable;
  }

  public List<String> getGenericTypes() {
    return genericTypes;
  }

  public Optional<String> getParent() {
    return isNotBlank(parent) ? of(parent) : empty();
  }

  public boolean isInterface() {
    return isInterface;
  }

  public boolean isFinal() {
    return isFinal;
  }

  public boolean isAbstract() {
    return isAbstract;
  }

  public boolean isMap() {
    return isMap;
  }

  public List<String> getImplementedInterfaces() {
    return implementedInterfaces;
  }

  public boolean hasDefaultConstructor() {
    return hasDefaultConstructor;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof ClassInformationAnnotation) {
      return Objects.equals(classname, ((ClassInformationAnnotation) obj).classname)
          && Objects.equals(genericTypes, ((ClassInformationAnnotation) obj).genericTypes);
    }

    return false;
  }

  @Override
  public int hashCode() {
    return Objects.hash(classname, genericTypes);
  }

  @Override
  public String toString() {
    return classname + (genericTypes.isEmpty() ? "" : genericTypes.stream().collect(joining(", ", "<", ">")));
  }
}
