/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java;

import org.junit.Before;
import org.junit.Test;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.metadata.java.api.handler.DefaultObjectFieldHandler;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.isOneOf;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.mule.metadata.api.builder.BaseTypeBuilder.create;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;

public class DefaultObjectFieldHandlerTestCase extends AbstractClassHandlerTestCase {

  @Before
  public void setup() {
    this.objectFieldHandler = new DefaultObjectFieldHandler();
    this.typeBuilder = create(JAVA).objectType();
  }

  @Test
  public void testNoFields() {
    handleFieldsOf(NoFields.class);
    List<ObjectFieldType> fields = getFields(typeBuilder);
    assertThat(fields, hasSize(0));
  }

  @Test
  public void testWithFields() {
    handleFieldsOf(WithFields.class);
    List<ObjectFieldType> fields = getFields(typeBuilder);
    assertThat(fields, hasSize(2));
    assertFieldType(fields.get(0), isOneOf(int.class.getName(), String.class.getName()));
  }

  @Test
  public void testWithParameterizedGeneric() {
    handleFieldsOf(WithParameterizedGeneric.class);
    List<ObjectFieldType> fields = getFields(typeBuilder);
    assertThat(fields, hasSize(1));
    assertFieldType(fields.get(0), is(Function.class.getName()));
    Optional<ClassInformationAnnotation> classInformation =
        fields.get(0).getValue().getAnnotation(ClassInformationAnnotation.class);
    assertThat(classInformation.isPresent(), is(true));
    assertThat(classInformation.get().getGenericTypes(), is(Arrays.asList(String.class.getName(), List.class.getName())));
  }

  @Test
  public void testOnlySetter() {
    handleFieldsOf(OnlySetter.class);
    List<ObjectFieldType> fields = getFields(typeBuilder);
    assertThat(fields, hasSize(1));
    assertFieldType(fields.get(0), is(Integer.class.getName()));
  }

  private class NoFields {

    String field;
  }

  private class WithFields {

    String field;
    int anotherField;

    public String getField() {
      return field;
    }

    public void setField(String field) {
      this.field = field;
    }

    public void setAnotherField(int anotherField) {
      this.anotherField = anotherField;
    }

    public int getAnotherField() {
      return anotherField;
    }
  }

  private class OnlySetter {

    Integer field;

    public void setField(Integer field) {
      this.field = field;
    }
  }

  private class WithParameterizedGeneric {

    Function<String, List<String>> parameterizedField;

    public Function<String, List<String>> getParameterizedField() {
      return parameterizedField;
    }

    public void setParameterizedField(Function<String, List<String>> parameterizedField) {
      this.parameterizedField = parameterizedField;
    }
  }
}
