/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api;

import static org.mule.metadata.api.utils.MetadataTypeUtils.addTypeAlias;
import static org.mule.metadata.java.api.utils.ClassUtils.getInnerClassName;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.java.api.handler.DefaultTypeHandlerManagerFactory;
import org.mule.metadata.java.api.handler.TypeHandlerManager;
import org.mule.metadata.java.api.handler.TypeHandlerManagerFactory;
import org.mule.metadata.java.api.utils.ParsingContext;

import java.lang.ref.WeakReference;
import java.lang.reflect.Type;
import java.util.Optional;

/**
 * Represents a java type loader system where you can load the definition of a class by its name.
 */
public class JavaTypeLoader implements ClassTypeLoader {

  public static MetadataFormat JAVA = MetadataFormat.JAVA;

  private final WeakReference<ClassLoader> classLoader;
  private final TypeHandlerManagerFactory typeHandlerManagerFactory;

  public JavaTypeLoader(ClassLoader classLoader) {
    this(classLoader, new DefaultTypeHandlerManagerFactory());
  }

  public JavaTypeLoader(ClassLoader classLoader, TypeHandlerManagerFactory typeHandlerManagerFactory) {
    if (classLoader == null) {
      throw new IllegalArgumentException("classloader cannot be null");
    }

    if (typeHandlerManagerFactory == null) {
      throw new IllegalArgumentException("typeHandlerManagerFactory cannot be null");
    }

    this.classLoader = new WeakReference<>(classLoader);
    this.typeHandlerManagerFactory = typeHandlerManagerFactory;
  }

  @Override
  public Optional<MetadataType> load(String identifier) {
    return load(identifier, null);
  }

  @Override
  public Optional<MetadataType> load(String identifier, String typeAlias) {
    if (void.class.getName().equals(identifier) || Void.class.getName().equals(identifier)) {
      return Optional.of(createTypeBuilder().voidType().build());
    }

    Class<?> clazz;
    try {
      clazz = classLoader.get().loadClass(identifier);
    } catch (ClassNotFoundException e) {
      try {
        clazz = classLoader.get().loadClass(getInnerClassName(identifier));
      } catch (ClassNotFoundException innerClassNotFound) {
        return Optional.empty();
      }
    }

    return Optional.of(load(clazz, typeAlias));
  }

  @Override
  public MetadataType load(Type type) {
    return load(type, null);
  }

  public MetadataType load(Type type, String typeAlias) {
    final BaseTypeBuilder typeBuilder = createTypeBuilder();
    if (void.class.equals(type) || Void.class.equals(type)) {
      return typeBuilder.voidType().build();
    }

    final TypeHandlerManager typeHandlerManager = typeHandlerManagerFactory.createTypeHandlerManager();
    typeHandlerManager.handle(type, new ParsingContext(), typeBuilder);
    addTypeAlias(typeBuilder, typeAlias);
    return typeBuilder.build();
  }

  @Override
  public ClassLoader getClassLoader() {
    return classLoader.get();
  }

  private BaseTypeBuilder createTypeBuilder() {
    return BaseTypeBuilder.create(JAVA);
  }
}
