/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.utils;

import java.lang.reflect.*;
import java.util.*;

public class TypeResolver {

  /**
   * Answers all super interfaces that the class implements, including transitively inherited ones.
   */
  public static Class<?>[] getSuperInterfaces(Class<?> type) {

    if (!type.isInterface()) {
      throw new IllegalArgumentException("only interfaces allowed");
    }
    final List<Class<?>> superInterfaces = new LinkedList<>();
    collectSuperInterfaces(type, superInterfaces);
    return superInterfaces.toArray(new Class[superInterfaces.size()]);
  }

  private static void collectSuperInterfaces(Class<?> type,
                                             List<Class<?>> superInterfaces) {

    for (Class<?> superInterface : type.getInterfaces()) {
      superInterfaces.add(superInterface);
      collectSuperInterfaces(superInterface, superInterfaces);
    }
  }

  /**
   * Answers immediate parameterized super type that is causing the type to be instance of given declaredType.
   *
   */
  public static ParameterizedType getGenericSuperclass(Class<?> type, Class<?> declaredType) {

    final List<Type> types = new ArrayList<>();
    if (type.getGenericSuperclass() != null) {
      types.add(type.getGenericSuperclass());
    }
    types.addAll(Arrays.asList(type.getGenericInterfaces()));

    for (Type superType : types) {
      if (superType instanceof ParameterizedType) {
        final ParameterizedType parameterizedType = (ParameterizedType) superType;
        final Type raw = parameterizedType.getRawType();
        if (raw instanceof Class<?>) {
          if (declaredType.isAssignableFrom((Class<?>) raw)) {
            return parameterizedType;
          }
        }
      }
    }
    return null;
  }

  /**
   * Resolves type variables in given class.
   */
  public static Map<TypeVariable<?>, Type> resolveVariables(Type type) {

    /*
     * only Class or ParameterizedType
     */
    if (!(type instanceof Class<?> || type instanceof ParameterizedType)) {
      throw new IllegalArgumentException("Invalid type " + type);
    }
    final Map<TypeVariable<?>, Type> map = new HashMap<>();
    final Class<?> raw = erase(type);
    final TypeVariable<?>[] formal = raw.getTypeParameters();
    final Type[] actual = type instanceof Class<?>
        ? formal
        : ((ParameterizedType) type).getActualTypeArguments();

    for (int i = 0; i < formal.length; ++i) {
      map.put(formal[i], actual[i]);
    }

    if (raw.getGenericSuperclass() != null) {
      map.putAll(resolveVariables(raw.getGenericSuperclass()));
    }
    for (Type genericInterface : raw.getGenericInterfaces()) {
      map.putAll(resolveVariables(genericInterface));
    }

    return map;
  }

  /**
   * Performs type erasure for given type.
   *
   */
  public static Class<?> erase(Type type) {

    if (type instanceof Class<?>) {
      return (Class<?>) type;
    }
    if (type instanceof ParameterizedType) {
      ParameterizedType parameterizedType = (ParameterizedType) type;
      return erase(parameterizedType.getRawType());
    }
    if (type instanceof TypeVariable<?>) {
      TypeVariable<?> typeVar = (TypeVariable<?>) type;
      Type[] bounds = typeVar.getBounds();
      if (bounds.length > 0) {
        return erase(bounds[0]);
      } else {
        return Object.class;
      }
    }
    if (type instanceof WildcardType) {
      WildcardType wildType = (WildcardType) type;
      Type[] bounds = wildType.getUpperBounds();
      if (bounds.length > 1) {
        return erase(bounds[0]);
      } else {
        return Object.class;
      }
    }
    if (type instanceof GenericArrayType) {
      GenericArrayType genericArray = (GenericArrayType) type;
      Class<?> componentClass = erase(genericArray
          .getGenericComponentType());
      return Array.newInstance(componentClass, 0).getClass();
    }
    throw new IllegalArgumentException("Unknown type " + type);
  }

}
