/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.mule.metadata.api.builder.BaseTypeBuilder.create;
import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import static org.mule.metadata.java.api.handler.TypeHandlerManager.createDefault;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.java.api.handler.DefaultObjectFieldHandler;
import org.mule.metadata.java.api.utils.ParsingContext;
import org.mule.metadata.java.internal.handler.MapClassHandler;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

public class MapClassHandlerTestCase extends AbstractClassHandlerTestCase {

  private MapClassHandler handler;

  @Before
  public void setup() {
    this.handler = new MapClassHandler();
    this.objectFieldHandler = new DefaultObjectFieldHandler();
    this.typeBuilder = create(JAVA).objectType();
  }

  @Test
  public void testInheritedGenericsFromInterface() {
    assertMapTypes(MyMap.class, WithCustomMap.class, StringType.class);
  }

  @Test
  public void testRawMap() {
    assertMapTypes(Map.class, WithRawMap.class, AnyType.class);
  }

  @Test
  public void testImplementsRawMap() {
    assertMapTypes(RawMapImplementation.class, WithRawMapImplementation.class, AnyType.class);
  }

  @Test
  public void testExtendsFromRawMapImplementation() {
    assertMapTypes(ExtendsFromRawMapImplementation.class, WithExtendsFromRawMapImplementation.class, AnyType.class);
  }

  @Test
  public void testGenericsResolvedInMapInterface() {
    assertMapTypes(MyExtraCustomMap.class, WithExtraCustomMap.class, StringType.class);
  }

  @Test
  public void testGenericsResolvedInSuperClass() {
    assertMapTypes(MapB.class, WithMapB.class, StringType.class);
  }

  @Test
  public void testGenericResolvedInClassWithMultipleResolutions() {
    assertMapTypes(ImplementationB.class, WithImplementationB.class, StringType.class);
  }

  @Test
  public void testWildcardResolvedInClassWithMultipleResolutions() {
    assertMapTypes(ImplementationB.class, WithImplementationBWithWildCard.class, StringType.class);
  }

  @Test
  public void testClassWithoutGeneric() {
    assertMapTypes(ImplementationB.class, WithImplementationBWithoutGeneric.class, AnyType.class);
  }

  @Test
  public void testFieldWithoutGenericResolvedButWildcardBoundInClass() {
    assertMapTypes(ImplementationC.class, WithImplementationCWithoutGeneric.class, StringType.class);
  }

  @Test
  public void testMapOfObjectInClass() {
    assertMapTypes(Map.class, WithMapOfObject.class, ObjectType.class);
  }

  @Test
  public void testMapImplementationOfObjectInClass() {
    assertMapTypes(MapA.class, WithMapImplementationOfObject.class, ObjectType.class);
  }

  private void assertMapTypes(Class<? extends Map> mapType, Class containerType, Class valueType) {
    assertThat(handler.handles(mapType), is(true));
    objectFieldHandler.handleFields(containerType, createDefault(), new ParsingContext(), typeBuilder);
    List<ObjectFieldType> fields = getFields(typeBuilder);
    assertThat(fields.isEmpty(), is(false));
    ObjectType field = (ObjectType) fields.get(0).getValue();
    assertThat(field.isOpen(), is(true));
    assertThat(field.getOpenRestriction().get(), instanceOf(valueType));
  }

  private class MyExtraCustomMap extends MyMap {
  }

  private class MyMap implements MyMapInterface {

    @Override
    public int size() {
      return 0;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }

    @Override
    public boolean containsKey(Object key) {
      return false;
    }

    @Override
    public boolean containsValue(Object value) {
      return false;
    }

    @Override
    public String get(Object key) {
      return null;
    }

    @Override
    public String put(String key, String value) {
      return null;
    }

    @Override
    public String remove(Object key) {
      return null;
    }

    @Override
    public void putAll(Map<? extends String, ? extends String> m) {

    }

    @Override
    public void clear() {

    }

    @Override
    public Set<String> keySet() {
      return null;
    }

    @Override
    public Collection<String> values() {
      return null;
    }

    @Override
    public Set<Entry<String, String>> entrySet() {
      return null;
    }
  }

  private interface MyMapInterface extends Map<String, String> {
  }

  private class MapA<K, V> implements Map<K, V> {

    @Override
    public int size() {
      return 0;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }

    @Override
    public boolean containsKey(Object key) {
      return false;
    }

    @Override
    public boolean containsValue(Object value) {
      return false;
    }

    @Override
    public V get(Object key) {
      return null;
    }

    @Override
    public V put(K key, V value) {
      return null;
    }

    @Override
    public V remove(Object key) {
      return null;
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {

    }

    @Override
    public void clear() {

    }

    @Override
    public Set<K> keySet() {
      return null;
    }

    @Override
    public Collection<V> values() {
      return null;
    }

    @Override
    public Set<Entry<K, V>> entrySet() {
      return null;
    }
  }

  private class ImplementationA<R, T, S> implements InterfaceA<R, S> {

    @Override
    public int size() {
      return 0;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }

    @Override
    public boolean containsKey(Object key) {
      return false;
    }

    @Override
    public boolean containsValue(Object value) {
      return false;
    }

    @Override
    public R get(Object key) {
      return null;
    }

    @Override
    public R put(Integer key, R value) {
      return null;
    }

    @Override
    public R remove(Object key) {
      return null;
    }

    @Override
    public void putAll(Map<? extends Integer, ? extends R> m) {

    }

    @Override
    public void clear() {

    }

    @Override
    public Set<Integer> keySet() {
      return null;
    }

    @Override
    public Collection<R> values() {
      return null;
    }

    @Override
    public Set<Entry<Integer, R>> entrySet() {
      return null;
    }
  }

  private class RawMapImplementation<S, T> implements Map {

    @Override
    public int size() {
      return 0;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }

    @Override
    public boolean containsKey(Object key) {
      return false;
    }

    @Override
    public boolean containsValue(Object value) {
      return false;
    }

    @Override
    public Object get(Object key) {
      return null;
    }

    @Override
    public Object put(Object key, Object value) {
      return null;
    }

    @Override
    public Object remove(Object key) {
      return null;
    }

    @Override
    public void putAll(Map m) {

    }

    @Override
    public void clear() {

    }

    @Override
    public Set keySet() {
      return null;
    }

    @Override
    public Collection values() {
      return null;
    }

    @Override
    public Set<Entry> entrySet() {
      return null;
    }
  }

  private class ExtendsFromRawMapImplementation extends RawMapImplementation<String, String> {

  }

  private interface InterfaceA<T, S> extends Map<Integer, T> {

  }

  private class ImplementationB<K> extends ImplementationA<K, Integer, Integer> {

  }

  private class ImplementationC<F extends String> extends ImplementationB<F> {

  }

  private class MapB extends MapA<String, String> {
  }

  private class WithRawMap {

    Map simpleMap;

    public Map getSimpleMap() {
      return simpleMap;
    }

  }

  private class WithRawMapImplementation {

    RawMapImplementation<Integer, Object> rawMapImplementation;

    public Map getRawMapImplementation() {
      return rawMapImplementation;
    }

  }

  private class WithExtendsFromRawMapImplementation {

    ExtendsFromRawMapImplementation extendsFromRawMapImplementation;

    public Map getExtendsFromRawMapImplementation() {
      return extendsFromRawMapImplementation;
    }
  }

  private class WithCustomMap {

    MyMap customMap;

    public MyMap getCustomMap() {
      return customMap;
    }

    public void setCustomMap(MyMap customMap) {
      this.customMap = customMap;
    }
  }

  private class WithExtraCustomMap {

    MyExtraCustomMap customMap;

    public MyMap getCustomMap() {
      return customMap;
    }

    public void setCustomMap(MyExtraCustomMap customMap) {
      this.customMap = customMap;
    }
  }

  private class WithMapB {

    MapB customMap;

    public MapB getCustomMap() {
      return customMap;
    }

    public void setCustomMap(MapB customMap) {
      this.customMap = customMap;
    }
  }

  private class WithImplementationB {

    ImplementationB<String> customMap;

    public ImplementationB<String> getCustomMap() {
      return customMap;
    }

    public void setCustomMap(ImplementationB<String> customMap) {
      this.customMap = customMap;
    }
  }

  private class WithImplementationBWithWildCard {

    ImplementationB<? extends String> customMap;

    public ImplementationB<? extends String> getCustomMap() {
      return customMap;
    }

    public void setCustomMap(ImplementationB<String> customMap) {
      this.customMap = customMap;
    }
  }

  private class WithImplementationBWithoutGeneric {

    ImplementationB customMap;

    public ImplementationB getCustomMap() {
      return customMap;
    }

    public void setCustomMap(ImplementationB customMap) {
      this.customMap = customMap;
    }
  }

  private class WithImplementationCWithoutGeneric {

    ImplementationC customMap;

    public ImplementationC getCustomMap() {
      return customMap;
    }

    public void setCustomMap(ImplementationC customMap) {
      this.customMap = customMap;
    }
  }

  private class WithMapOfObject {

    Map<String, Object> mapOfObject;

    public Map<String, Object> getMapOfObject() {
      return mapOfObject;
    }

    public void setMapOfObject(Map<String, Object> mapOfObject) {
      this.mapOfObject = mapOfObject;
    }
  }


  private class WithMapImplementationOfObject {

    MapA<String, Object> mapAOfObject;

    public MapA<String, Object> getMapOfObject() {
      return mapAOfObject;
    }

    public void setMapOfObject(MapA<String, Object> mapOfObject) {
      this.mapAOfObject = mapOfObject;
    }
  }
}
