/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.utils;

import static java.util.Collections.unmodifiableMap;

import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.internal.utils.ParsingSubContext;

import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

public class ParsingContext {

  private final ExtensionParsingContext extensionParsingContext;
  private final Map<TypeVariable<?>, Type> resolvedVariables = new HashMap<>();
  private final Map<String, ParsingSubContext> subContexts = new HashMap<>();

  public ParsingContext() {
    this.extensionParsingContext = new ExtensionParsingContext();
  }

  public ParsingContext(ExtensionParsingContext extensionParsingContext) {
    this.extensionParsingContext = extensionParsingContext;
  }

  public Optional<TypeBuilder<?>> getTypeBuilder(Type type) {
    return extensionParsingContext.getTypeBuilder(type);
  }

  public void addTypeBuilder(Type type, TypeBuilder<?> typedObject) {
    extensionParsingContext.addTypeBuilder(type, typedObject);
  }

  public Map<TypeVariable<?>, Type> getResolvedVariables() {
    return unmodifiableMap(resolvedVariables);
  }

  public void addResolvedVariables(Map<TypeVariable<?>, Type> map) {
    for (Map.Entry<TypeVariable<?>, Type> entry : map.entrySet()) {
      if (resolvedVariables.containsKey(entry.getKey())) {
        // only replace if value is something usable
        if (!(entry.getValue() instanceof TypeVariable<?>)) {
          resolvedVariables.put(entry.getKey(), entry.getValue());
        }
      } else {
        resolvedVariables.put(entry.getKey(), entry.getValue());
      }
    }
  }

  /**
   * Provides an isolated context that can be retrieved at any time from this context.
   * 
   * @param subId the key of the isolated context to get. It it does not exist, a new one will be created.
   * @return an isolated context
   * 
   * @since 1.9
   */
  public ParsingContext getSubContext(String subId) {
    return subContexts.computeIfAbsent(subId, k -> new ParsingSubContext(k, this));
  }
}
