/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.api.handler;

import static org.mule.metadata.java.api.annotation.ClassInformationAnnotation.INTERNAL_METADATA_PACKAGE;

import static java.beans.Introspector.getBeanInfo;
import static java.util.stream.Stream.concat;

import org.mule.metadata.api.annotation.Accessibility;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.utils.ParsingContext;
import org.mule.metadata.java.api.utils.TypeResolver;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Type;
import java.util.Optional;
import java.util.stream.Stream;

public class DefaultObjectFieldHandler implements ObjectFieldHandler {

  @Override
  public void handleFields(Class<?> clazz,
                           TypeHandlerManager typeHandlerManager, ParsingContext context,
                           ObjectTypeBuilder builder) {
    if (!Object.class.equals(clazz) && !Void.class.equals(clazz) && !void.class.equals(clazz)) {
      try {
        loadProperties(clazz)
            // property class returns null
            // if there are getters with
            // no associated field.
            .filter(pd -> pd.getPropertyType() != null)
            .forEach(pd -> {
              final ObjectFieldTypeBuilder fieldBuilder = builder.addField();
              final String name = pd.getName();
              fieldBuilder.key(name);
              final Optional<Accessibility> visibility = getVisibility(pd);
              if (visibility.isPresent()) {
                fieldBuilder.accessibility(visibility.get());
              }
              final Type propertyType = getPropertyType(pd);
              // Check declarations if it was already defined
              final Optional<TypeBuilder<?>> typeBuilder = context.getTypeBuilder(propertyType);
              if (typeBuilder.isPresent()) {
                fieldBuilder.value(typeBuilder.get());
              } else {
                typeHandlerManager.handle(propertyType, context, fieldBuilder.value());
              }
            });
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
  }

  private Optional<Accessibility> getVisibility(PropertyDescriptor pd) {
    if (pd.getReadMethod() != null && pd.getWriteMethod() == null) {
      return Optional.of(Accessibility.READ_ONLY);
    } else if (pd.getReadMethod() == null && pd.getWriteMethod() != null) {
      return Optional.of(Accessibility.WRITE_ONLY);
    }
    return Optional.empty();
  }

  private Type getPropertyType(PropertyDescriptor pd) {
    final Type propertyType;
    if (pd.getReadMethod() != null) {
      propertyType = pd.getReadMethod().getGenericReturnType();
    } else {
      propertyType = pd.getWriteMethod().getParameterTypes()[0];
    }
    return propertyType;
  }

  private Stream<PropertyDescriptor> loadProperties(Class<?> clazz) throws IntrospectionException {
    Stream<PropertyDescriptor> propertyDescriptorsStream;
    if (clazz.isInterface()) {
      propertyDescriptorsStream = Stream.of(getBeanInfo(clazz).getPropertyDescriptors());
      for (Class<?> interfaceType : TypeResolver.getSuperInterfaces(clazz)) {
        propertyDescriptorsStream =
            concat(propertyDescriptorsStream, Stream.of(getBeanInfo(interfaceType).getPropertyDescriptors()));
      }
    } else {
      propertyDescriptorsStream = Stream.of(getBeanInfo(clazz, Object.class).getPropertyDescriptors());
    }
    return propertyDescriptorsStream
        .filter(pd -> {
          if (pd.getReadMethod() != null) {
            return !pd.getReadMethod().getDeclaringClass().getName().startsWith(INTERNAL_METADATA_PACKAGE);
          }
          if (pd.getWriteMethod() != null) {
            return !pd.getWriteMethod().getDeclaringClass().getName().startsWith(INTERNAL_METADATA_PACKAGE);
          }
          // Both methods being null means either the class is not a valid bean or that the bean has an indexed property
          // which we don't currently support.
          return false;
        });
  }
}
