/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.example;

import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataType;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

public class ArrayHandler implements JsonElementHandler {

  @Override
  public boolean handles(JsonElement jsonElement) {
    return jsonElement.isJsonArray();
  }

  @Override
  public TypeBuilder<?> handle(JsonElement jsonElement, BaseTypeBuilder root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {
    JsonArray array = (JsonArray) jsonElement;
    final ArrayTypeBuilder arrayMetadataBuilder = root.arrayType();

    parsingContext.getAnnotations().forEach(arrayMetadataBuilder::with);

    final JsonElement arraySample = getFirstChild(array);
    if (arraySample != null) {
      List<MetadataType> uniqueTypes = new ArrayList<>();
      Map<MetadataType, TypeBuilder<?>> builders = new LinkedHashMap<>();
      for (Iterator<JsonElement> it = array.iterator(); it.hasNext();) {
        JsonElement item = it.next();
        TypeBuilder<?> itemBuilder = handlerManager.handle(item, new ParsingContext());
        MetadataType type = itemBuilder.build();
        if (!builders.containsKey(type)) {
          builders.put(type, itemBuilder);
        }
      }
      // Create union or simple type
      if (builders.size() > 1) {
        UnionTypeBuilder unionTypeBuilder = arrayMetadataBuilder.of().unionType();
        builders.values().forEach(unionTypeBuilder::of);
      } else {
        arrayMetadataBuilder.of(handlerManager.handle(arraySample, new ParsingContext()));
      }
    } else {
      arrayMetadataBuilder.of().anyType();
    }
    return arrayMetadataBuilder;
  }

  private JsonElement getFirstChild(JsonArray array) {
    for (Iterator<JsonElement> i = array.iterator(); i.hasNext();) {
      return i.next();
    }
    return null;
  }
}
