/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json;

import org.everit.json.schema.Schema;
import org.everit.json.schema.loader.SchemaLoader;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.utils.MetadataTypeUtils;
import org.mule.metadata.json.handler.HandlerManager;
import org.mule.metadata.json.handler.ParsingContext;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Optional;

public class JsonTypeLoader implements TypeLoader {

  public static MetadataFormat JSON = MetadataFormat.JSON;
  private JsonSchemaLoader jsonSchemaLoader;


  public JsonTypeLoader(File schemaFile) {
    jsonSchemaLoader = new JsonSchemaFileLoader(schemaFile);
  }


  public JsonTypeLoader(String schemaData) {
    jsonSchemaLoader = new JsonSchemaStringLoader(schemaData);
  }

  private JsonSchemaLoader getJsonSchemaLoader() {
    return jsonSchemaLoader;
  }

  @Override
  public Optional<MetadataType> load(String identifier) {
    return load(identifier, null);
  }

  @Override
  public Optional<MetadataType> load(String identifier, String typeAlias) {
    final Schema jsonSchema = getJsonSchemaLoader().loadSchema();
    final TypeBuilder<?> typeBuilder = new HandlerManager().handle(jsonSchema, new ParsingContext());
    MetadataTypeUtils.addTypeAlias(typeBuilder, typeAlias);
    return Optional.of(typeBuilder.build());
  }

  private interface JsonSchemaLoader {

    Schema loadSchema();
  }


  private static class JsonSchemaFileLoader implements JsonSchemaLoader {

    private File schemaFile;

    public JsonSchemaFileLoader(File schemaFile) {
      this.schemaFile = schemaFile;
    }

    @Override
    public Schema loadSchema() {
      try {
        try (InputStream inputStream = new FileInputStream(schemaFile)) {
          final JSONObject rawSchema = new JSONObject(new JSONTokener(inputStream));
          return SchemaLoader.load(rawSchema);
        }
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }
  }


  private static class JsonSchemaStringLoader implements JsonSchemaLoader {

    private String schemaData;

    public JsonSchemaStringLoader(String schemaData) {
      this.schemaData = schemaData;
    }

    @Override
    public Schema loadSchema() {
      final JSONObject rawSchema = new JSONObject(new JSONTokener(schemaData));
      return SchemaLoader.load(rawSchema);
    }
  }
}
