/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.json.example.HandlerManager;
import org.mule.metadata.json.example.ParsingContext;

import com.google.gson.JsonElement;
import com.google.gson.JsonParser;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Optional;

import org.apache.commons.lang3.StringEscapeUtils;

import static org.mule.metadata.api.utils.MetadataTypeUtils.addTypeAlias;

public class JsonExampleTypeLoader implements TypeLoader {

  private Optional<File> exampleFile;
  private Optional<String> example;

  public JsonExampleTypeLoader(File exampleFile) {
    this.exampleFile = Optional.of(exampleFile);
    this.example = Optional.empty();
  }

  public JsonExampleTypeLoader(String example) {
    this.exampleFile = Optional.empty();
    this.example = Optional.of(example);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    try (Reader bufferedReader = buildReader()) {
      JsonElement root = new JsonParser().parse(bufferedReader);
      String content = root.toString();
      ExampleAnnotation example = new ExampleAnnotation(StringEscapeUtils.escapeJson(content));
      final TypeBuilder<?> typeBuilder = new HandlerManager().handle(root, new ParsingContext().withAnnotation(example));
      addTypeAlias(typeBuilder, typeAlias);
      return Optional.of(typeBuilder.build());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private Reader buildReader() throws FileNotFoundException {
    return exampleFile.isPresent() ? new FileReader(exampleFile.get())
        : new StringReader(example.orElseThrow(() -> new RuntimeException("No example provided.")));
  }
}
