/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.handler;

import org.everit.json.schema.ArraySchema;
import org.everit.json.schema.Schema;
import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TupleTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import java.util.List;

import static org.mule.metadata.api.utils.MetadataTypeUtils.isNotNull;
import static org.mule.metadata.internal.utils.StringUtils.isNotEmpty;

public class ArrayHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof ArraySchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {
    ArraySchema arraySchema = (ArraySchema) schema;

    if (arraySchema.getItemSchemas() != null && !arraySchema.getItemSchemas().isEmpty()) {
      final List<Schema> itemSchemas = arraySchema.getItemSchemas();
      final TupleTypeBuilder tupleType = root.tupleType();
      final String description = arraySchema.getDescription();
      if (isNotEmpty(description)) {
        tupleType.description(description);
      }

      final String title = arraySchema.getTitle();
      if (isNotEmpty(title)) {
        tupleType.label(title);
      }
      for (Schema itemSchema : itemSchemas) {
        tupleType.of(handlerManager.handle(itemSchema, parsingContext));
      }
      return tupleType;
    } else {

      final ArrayTypeBuilder arrayMetadataBuilder = root.arrayType();
      final String description = arraySchema.getDescription();
      if (isNotEmpty(description)) {
        arrayMetadataBuilder.description(description);
      }

      final String title = arraySchema.getTitle();
      if (isNotEmpty(title)) {
        arrayMetadataBuilder.label(title);
      }

      if (arraySchema.needsUniqueItems()) {
        arrayMetadataBuilder.uniques();
      }

      Integer min = null;
      Integer max = null;

      if (arraySchema.getMinItems() != null && arraySchema.getMinItems() > 0) {
        min = arraySchema.getMinItems();
      }

      if (arraySchema.getMaxItems() != null && arraySchema.getMaxItems() > 0) {
        max = arraySchema.getMaxItems();
      }

      if (isNotNull(min) || isNotNull(max)) {
        arrayMetadataBuilder.boundary(min, max);
      }

      final Schema arrayType = arraySchema.getAllItemSchema();
      if (arrayType != null) {
        arrayMetadataBuilder.of(handlerManager.handle(arrayType, parsingContext));
      } else {
        arrayMetadataBuilder.of().anyType();
      }
      return arrayMetadataBuilder;
    }
  }
}
