/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.example;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import org.mule.metadata.api.builder.ArrayTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;
import org.mule.metadata.api.model.MetadataType;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

public class ArrayHandler implements JsonElementHandler {

  @Override
  public boolean handles(JsonElement jsonElement) {
    return jsonElement.isJsonArray();
  }

  @Override
  public TypeBuilder<?> handle(JsonElement jsonElement, BaseTypeBuilder root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {
    JsonArray array = (JsonArray) jsonElement;
    final ArrayTypeBuilder arrayMetadataBuilder = root.arrayType();

    parsingContext.getAnnotations().forEach(arrayMetadataBuilder::with);

    final JsonElement arraySample = getFirstChild(array);
    if (arraySample != null) {
      Map<MetadataType, TypeBuilder<?>> builders = new LinkedHashMap<>();
      for (Iterator<JsonElement> it = array.iterator(); it.hasNext();) {
        JsonElement item = it.next();
        TypeBuilder<?> itemBuilder =
            handlerManager.handle(item, new ParsingContext(parsingContext.getHandlerConfiguration()));
        MetadataType type = itemBuilder.build();
        if (!builders.containsKey(type)) {
          builders.put(type, itemBuilder);
        }
      }
      // Create union or simple type
      if (builders.size() > 1) {
        if (parsingContext.getHandlerConfiguration().isMergeObjectUnionTypes() && allChildBuilderAreObjectBuilders(builders)) {
          //Merge the child elements of the array and process again.
          JsonElement mergedArrayExample = mergeExamples(array);
          arrayMetadataBuilder.of(handlerManager
              .handle(mergedArrayExample,
                      new ParsingContext(parsingContext.getHandlerConfiguration())));
        } else {
          UnionTypeBuilder unionTypeBuilder = arrayMetadataBuilder.of().unionType();
          builders.values().forEach(unionTypeBuilder::of);
        }
      } else {
        arrayMetadataBuilder.of(handlerManager
            .handle(arraySample, new ParsingContext(parsingContext.getHandlerConfiguration())));
      }
    } else {
      arrayMetadataBuilder.of().anyType();
    }
    return arrayMetadataBuilder;
  }

  private boolean allChildBuilderAreObjectBuilders(Map<MetadataType, TypeBuilder<?>> builders) {
    return builders.values().stream().allMatch(ObjectTypeBuilder.class::isInstance);
  }

  private JsonElement getFirstChild(JsonArray array) {
    for (Iterator<JsonElement> i = array.iterator(); i.hasNext();) {
      return i.next();
    }
    return null;
  }

  private JsonElement mergeExamples(JsonArray array) {
    JsonObject innerObject = new JsonObject();
    Set<String> properties = new HashSet<>();
    Iterator<JsonElement> it = array.iterator();
    while (it.hasNext()) {
      JsonObject item = (JsonObject) it.next();
      item.entrySet().stream().forEach(field -> {
        String property = field.getKey();
        if (!properties.contains(property)) {
          properties.add(property);
          innerObject.add(property, field.getValue());
        }
      });
    }
    return innerObject;
  }
}
