/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.persistence.type.adapter.EnumTypeAnnotationTypeAdapter;
import org.mule.metadata.persistence.type.adapter.OnlyOneFieldTypeAdapterFactory;
import org.mule.metadata.persistence.type.adapter.OptionalTypeAdapterFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonWriter;

import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Serializer for {@link TypeAnnotation}.
 *
 * @param <T> {@link TypeAnnotation} implementation
 * @since 1.0
 */
public final class TypeAnnotationSerializer<T extends TypeAnnotation> {

  private static final Logger LOGGER = LoggerFactory.getLogger(TypeAnnotationSerializer.class);
  private static final TypeAnnotationNameClassMapping annotationsNameClassMapping = new TypeAnnotationNameClassMapping();
  private static final String ERROR_DESERIALIZING_TYPE_ANNOTATION = "Error deserializing TypeAnnotation '%s'.";
  private final Gson gson;

  public TypeAnnotationSerializer() {
    gson = new GsonBuilder()
        .registerTypeAdapterFactory(new OptionalTypeAdapterFactory())
        .registerTypeAdapter(EnumAnnotation.class, new EnumTypeAnnotationTypeAdapter())
        .registerTypeAdapterFactory(new OnlyOneFieldTypeAdapterFactory(AccessibilityAnnotation.class,
                                                                       LabelAnnotation.class,
                                                                       DefaultValueAnnotation.class,
                                                                       TypeIdAnnotation.class,
                                                                       RegexPatternAnnotation.class,
                                                                       ExampleAnnotation.class,
                                                                       DefaultValueAnnotation.class,
                                                                       RegexPatternAnnotation.class))
        .create();
  }

  /**
   * Deserializes a {@link TypeAnnotation} implementation.
   *
   * @param typeAnnotationName name of the {@link TypeAnnotation} that match with {@link TypeAnnotation#getName()} or
   *                           their full qualifier class name
   * @param json               {@link String} representation of the serialized {@link TypeAnnotation}
   * @return An {@link Optional<TypeAnnotation>} with the deserialized {@link TypeAnnotation} in case that the deserialization
   * could not be done, {@link Optional#empty()} will be returned
   */
  public Optional<T> deserialize(String typeAnnotationName, String json) {
    Optional<T> optionalSerialization = Optional.empty();
    try {
      final Optional<Class<? extends TypeAnnotation>> annotationClass = getAnnotationClass(typeAnnotationName);
      if (annotationClass.isPresent()) {
        optionalSerialization = Optional.of((T) gson.getAdapter(annotationClass.get()).fromJson(json));
      }
    } catch (Exception e) {
      LOGGER.error(String.format(ERROR_DESERIALIZING_TYPE_ANNOTATION, typeAnnotationName), e);
    }

    return optionalSerialization;
  }

  /**
   * Serializes a {@link TypeAnnotation} implementation.
   *
   * @param jsonWriter     {@link JsonWriter} to write the serialized {@link TypeAnnotation}
   * @param typeAnnotation {@link TypeAnnotation} implementation to serialized
   */
  void serialize(JsonWriter jsonWriter, T typeAnnotation) {
    gson.toJson(typeAnnotation, new TypeToken<T>() {}.getType(), jsonWriter);
  }

  private Optional<Class<? extends TypeAnnotation>> getAnnotationClass(String annotationName) {
    final Map<String, Class<? extends TypeAnnotation>> nameClassMapping = annotationsNameClassMapping.getNameClassMapping();
    Optional<Class<? extends TypeAnnotation>> optionalClass = Optional.empty();

    if (nameClassMapping.containsKey(annotationName)) {
      optionalClass = Optional.of(nameClassMapping.get(annotationName));
    } else {
      try {
        optionalClass = Optional.of((Class<? extends TypeAnnotation>) Class.forName(annotationName));
      } catch (Exception e) {
        LOGGER.error(String.format(ERROR_DESERIALIZING_TYPE_ANNOTATION, annotationName), e);
      }
    }
    return optionalClass;
  }
}
