/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import org.mule.metadata.api.annotation.AccessibilityAnnotation;
import org.mule.metadata.api.annotation.DefaultValueAnnotation;
import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.ExampleAnnotation;
import org.mule.metadata.api.annotation.FieldOccurrenceAnnotation;
import org.mule.metadata.api.annotation.IntAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.annotation.NumberRangeAnnotation;
import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAliasAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.annotation.UniquesItemsAnnotation;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.metadata.persistence.api.TypeAnnotationSerializerExtenderFallback;
import org.mule.metadata.persistence.api.TypeAnnotationSerializerExtender;

import com.google.common.collect.HashBiMap;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.ServiceLoader;

/**
 * Factory class for instances of {@link TypeAnnotationSerializer} which have been enriched with SPI discovered implementations
 * of {@link TypeAnnotationSerializer}
 * <p>
 * By default this mapping provides all the out of the box {@link TypeAnnotation} implementation like {@link LabelAnnotation},
 * {@link DefaultValueAnnotation}, etc, but this mapping is extensible to provide more mappings using SPI (Service Provider
 * Interface) loading defined {@link TypeAnnotationSerializerExtender} implementations.
 *
 * @see TypeAnnotationSerializerExtender
 * @since 1.0
 */
public final class TypeAnnotationSerializerFactory {

  private static TypeAnnotationSerializerFactory INSTANCE = new TypeAnnotationSerializerFactory();

  public static TypeAnnotationSerializerFactory getInstance() {
    return INSTANCE;
  }

  private final HashBiMap<String, Class<? extends TypeAnnotation>> stringClassHashMap = HashBiMap.create();
  private final Collection<Object> additionalFeatures = new LinkedList<>();
  private final TypeAnnotationSerializerExtenderFallback typeAnnotationSerializerExtenderFallback;

  private TypeAnnotationSerializerFactory() {
    stringClassHashMap.put(LabelAnnotation.NAME, LabelAnnotation.class);
    stringClassHashMap.put(LengthAnnotation.NAME, LengthAnnotation.class);
    stringClassHashMap.put(AccessibilityAnnotation.NAME, AccessibilityAnnotation.class);
    stringClassHashMap.put(DefaultValueAnnotation.NAME, DefaultValueAnnotation.class);
    stringClassHashMap.put(DescriptionAnnotation.NAME, DescriptionAnnotation.class);
    stringClassHashMap.put(EnumAnnotation.NAME, EnumAnnotation.class);
    stringClassHashMap.put(ExampleAnnotation.NAME, ExampleAnnotation.class);
    stringClassHashMap.put(FieldOccurrenceAnnotation.NAME, FieldOccurrenceAnnotation.class);
    stringClassHashMap.put(IntAnnotation.NAME, IntAnnotation.class);
    stringClassHashMap.put(NumberRangeAnnotation.NAME, NumberRangeAnnotation.class);
    stringClassHashMap.put(RegexPatternAnnotation.NAME, RegexPatternAnnotation.class);
    stringClassHashMap.put(TypeIdAnnotation.NAME, TypeIdAnnotation.class);
    stringClassHashMap.put(UniquesItemsAnnotation.NAME, UniquesItemsAnnotation.class);
    stringClassHashMap.put(ClassInformationAnnotation.NAME, ClassInformationAnnotation.class);
    stringClassHashMap.put(TypeAliasAnnotation.NAME, TypeAliasAnnotation.class);

    ServiceLoader<TypeAnnotationSerializerExtender> extenders = ServiceLoader.load(TypeAnnotationSerializerExtender.class);
    extenders.forEach(extender -> {
      stringClassHashMap.putAll(extender.getNameClassMapping());
      additionalFeatures.addAll(extender.getAdditionalFeatures());
    });

    Iterator<TypeAnnotationSerializerExtenderFallback> fallbackIterator =
        ServiceLoader.load(TypeAnnotationSerializerExtenderFallback.class).iterator();
    if (fallbackIterator.hasNext()) {
      typeAnnotationSerializerExtenderFallback = fallbackIterator.next();
    } else {
      typeAnnotationSerializerExtenderFallback = null;
    }
  }

  public TypeAnnotationSerializer getTypeAnnotationSerializer() {
    return new TypeAnnotationSerializer(stringClassHashMap, additionalFeatures, typeAnnotationSerializerExtenderFallback);
  }
}
