/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import static java.util.stream.Collectors.toList;
import static org.mule.metadata.persistence.MetadataTypeConstants.ANNOTATIONS;
import static org.mule.metadata.persistence.MetadataTypeConstants.ANY;
import static org.mule.metadata.persistence.MetadataTypeConstants.ARRAY;
import static org.mule.metadata.persistence.MetadataTypeConstants.ATTRIBUTES;
import static org.mule.metadata.persistence.MetadataTypeConstants.BINARY;
import static org.mule.metadata.persistence.MetadataTypeConstants.BOOLEAN;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE;
import static org.mule.metadata.persistence.MetadataTypeConstants.DATE_TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.FIELDS;
import static org.mule.metadata.persistence.MetadataTypeConstants.FORMAT;
import static org.mule.metadata.persistence.MetadataTypeConstants.FUNCTION;
import static org.mule.metadata.persistence.MetadataTypeConstants.ID;
import static org.mule.metadata.persistence.MetadataTypeConstants.INTERSECTION;
import static org.mule.metadata.persistence.MetadataTypeConstants.ITEM;
import static org.mule.metadata.persistence.MetadataTypeConstants.KEY;
import static org.mule.metadata.persistence.MetadataTypeConstants.LABEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.MODEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NAME;
import static org.mule.metadata.persistence.MetadataTypeConstants.NOTHING;
import static org.mule.metadata.persistence.MetadataTypeConstants.NULL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NUMBER;
import static org.mule.metadata.persistence.MetadataTypeConstants.OBJECT;
import static org.mule.metadata.persistence.MetadataTypeConstants.OF;
import static org.mule.metadata.persistence.MetadataTypeConstants.OPEN;
import static org.mule.metadata.persistence.MetadataTypeConstants.ORDERED;
import static org.mule.metadata.persistence.MetadataTypeConstants.PARAMETERS;
import static org.mule.metadata.persistence.MetadataTypeConstants.RECURSION_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.REPEATED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REQUIRED;
import static org.mule.metadata.persistence.MetadataTypeConstants.RETURN_TYPE;
import static org.mule.metadata.persistence.MetadataTypeConstants.STRING;
import static org.mule.metadata.persistence.MetadataTypeConstants.TIME;
import static org.mule.metadata.persistence.MetadataTypeConstants.TRUE;
import static org.mule.metadata.persistence.MetadataTypeConstants.TUPLE;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;
import static org.mule.metadata.persistence.MetadataTypeConstants.UNION;
import static org.mule.metadata.persistence.MetadataTypeConstants.VALID_MIME_TYPES;
import static org.mule.metadata.persistence.MetadataTypeConstants.VOID;
import static org.mule.metadata.persistence.MetadataTypeConstants.commonMetadataFormats;
import org.mule.metadata.api.TypeWriter;
import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.AnyType;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.AttributeFieldType;
import org.mule.metadata.api.model.AttributeKeyType;
import org.mule.metadata.api.model.BinaryType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.FunctionParameter;
import org.mule.metadata.api.model.FunctionType;
import org.mule.metadata.api.model.IntersectionType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NothingType;
import org.mule.metadata.api.model.NullType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.SimpleType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.api.model.TimeType;
import org.mule.metadata.api.model.TupleType;
import org.mule.metadata.api.model.UnionType;
import org.mule.metadata.api.model.VoidType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 * {@link TypeWriter} that serializes {@link MetadataType} objects into a readable and processable JSON file, that
 * can be deserialized again into a {@link MetadataType} using {@link JsonMetadataTypeLoader}.
 * This class is not Thread-safe.
 *
 * @since 1.0
 */
public class JsonMetadataTypeWriter extends MetadataTypeVisitor implements TypeWriter {

  private static final String UNEXPECTED_ERROR_OCCURRED_SERIALIZING = "Unexpected error occurred serializing %s";
  private static final String INDENT_TAB = "  ";
  private final ObjectTypeReferenceHandler referenceHandler;
  private boolean prettyPrint;
  private JsonWriter writer;
  private Stack<MetadataType> typeStack;
  private TypeAnnotationSerializer typeAnnotationSerializer =
      TypeAnnotationSerializerFactory.getInstance().getTypeAnnotationSerializer();

  /**
   * Creates a new instance which serializes all types explicitly without
   * handling references
   */
  public JsonMetadataTypeWriter() {
    this(new NullObjectTypeReferenceHandler());
  }

  /**
   * Creates a new instance which handles type references through the given
   * {@code referenceHandler}
   *
   * @param referenceHandler an {@link ObjectTypeReferenceHandler}
   */
  public JsonMetadataTypeWriter(ObjectTypeReferenceHandler referenceHandler) {
    this.referenceHandler = referenceHandler;
  }

  /**
   * Given a {@link MetadataType}, serializes it into JSON format, the result of the serialization will be the output
   * of the operation as a {@link String}
   *
   * @param structure {@link MetadataType} to serialize.
   * @throws MetadataSerializingException
   */
  @Override
  public String toString(MetadataType structure) {
    try {
      StringWriter out = new StringWriter();
      writer = new JsonWriter(out);
      if (prettyPrint) {
        writer.setIndent(INDENT_TAB);
      }
      write(structure, writer);
      return out.toString();
    } catch (IOException e) {
      throw new MetadataSerializingException(String.format(UNEXPECTED_ERROR_OCCURRED_SERIALIZING, "MetadataType"), e);
    }
  }

  /**
   * Given a {@link MetadataType}, serializes it into a JSON format, the result of the serialization will be written
   * by the {@param jsonWriter}.
   *
   * @param metadataType {@link MetadataType} to serialize.
   * @param jsonWriter   {@link JsonWriter} which will be used to write the serialized {@link MetadataType}
   * @throws IOException
   */
  public void write(MetadataType metadataType, JsonWriter jsonWriter) throws IOException {
    writer = jsonWriter;
    typeStack = new Stack<>();
    write(metadataType);
  }

  /**
   * Changes the value of PrettyPrint property of the {@link JsonMetadataTypeWriter}. If it is enabled, when {@link #toString(MetadataType)}
   * is executed the output {@link String} will be printed in a human readable format, otherwise, by default, the JSON
   * will be printed in a compact and more performable format.
   *
   * @param prettyPrint boolean value indicating if the writer should pretty print the JSON
   * @return the configured {@link JsonMetadataTypeWriter} instance
   */
  public JsonMetadataTypeWriter setPrettyPrint(boolean prettyPrint) {
    this.prettyPrint = prettyPrint;
    return this;
  }

  private void write(MetadataType type)
      throws IOException {

    //Only complex types can be recursive
    if (!(type instanceof SimpleType) && typeStack.contains(type)) {
      //Recursion detected
      final int indexOf = typeStack.indexOf(type);
      final int reference = typeStack.size() - indexOf;
      String ref = RECURSION_FLAG;
      for (int i = 0; i < reference; i++) {
        if (i > 0) {
          ref = ref + "/";
        }
        ref = ref + "..";
      }
      writer.value(ref);
    } else {
      writer.beginObject();
      final MetadataFormat metadataFormat = type.getMetadataFormat();
      if (typeStack.isEmpty() || metadataFormat != typeStack.peek().getMetadataFormat()) {
        writer.name(FORMAT);
        if (commonMetadataFormats.contains(metadataFormat)) {
          writer.value(metadataFormat.getId());
        } else {
          writer.beginObject();
          writer.name(ID).value(metadataFormat.getId());
          if (metadataFormat.getLabel().isPresent()) {
            writer.name(LABEL).value(metadataFormat.getLabel().get());
          }
          writer.name(VALID_MIME_TYPES);
          writer.beginArray();
          for (String s : metadataFormat.getValidMimeTypes()) {
            writer.value(s);
          }
          writer.endArray();
          writer.endObject();
        }

      }
      typeStack.push(type);
      type.accept(this);
      writer.endObject();
      typeStack.pop();
    }
  }


  @Override
  public void visitAnyType(AnyType anyType) {
    writeType(anyType, ANY);
  }

  @Override
  public void visitArrayType(ArrayType arrayType) {
    writeType(arrayType, ARRAY);

    try {
      writer.name(ITEM);
      write(arrayType.getType());
    } catch (IOException e) {
      throw new MetadataSerializingException("Unexpected error occurred serializing ObjectType", e);
    }
  }

  @Override
  public void visitBinaryType(BinaryType binaryType) {
    writeType(binaryType, BINARY);
  }

  @Override
  public void visitBoolean(BooleanType booleanType) {
    writeType(booleanType, BOOLEAN);
  }

  @Override
  public void visitDateTime(DateTimeType dateTimeType) {
    writeType(dateTimeType, DATE_TIME);
  }

  @Override
  public void visitDate(DateType dateType) {
    writeType(dateType, DATE);
  }


  @Override
  public void visitNull(NullType nullType) {
    writeType(nullType, NULL);
  }

  @Override
  public void visitNothing(NothingType nothingType) {
    writeType(nothingType, NOTHING);
  }

  @Override
  public void visitVoid(VoidType voidType) {
    writeType(voidType, VOID);
  }

  @Override
  public void visitNumber(NumberType numberType) {
    writeType(numberType, NUMBER);
  }

  @Override
  public void visitObject(ObjectType objectType) {
    if (referenceHandler.writeReference(objectType, writer).isPresent()) {
      return;
    }

    writeType(objectType, OBJECT);
    final Collection<ObjectFieldType> fields = objectType.getFields();
    try {
      if (objectType.isOrdered()) {
        writer.name(ORDERED).value(true);
      }
      if (objectType.isOpen()) {
        writer.name(OPEN);
        write(objectType.getOpenRestriction().get());
      }
      writer.name(FIELDS);
      writer.beginArray();

      for (ObjectFieldType field : fields) {
        writer.beginObject();
        final ObjectKeyType key = field.getKey();
        final Collection<TypeAnnotation> keyAnnotations = key.getAnnotations();

        final HashMap<String, String> stringObjectHashMap = new HashMap<>();
        if (field.isRequired()) {
          stringObjectHashMap.put(REQUIRED, TRUE);
        }
        if (field.isRepeated()) {
          stringObjectHashMap.put(REPEATED, TRUE);
        }
        createKeyObject(keyAnnotations, field.getKey(), stringObjectHashMap);
        writer.name(MODEL);
        write(field.getValue());
        writeAnnotations(field.getAnnotations());
        writer.endObject();
      }
      writer.endArray();
    } catch (IOException e) {
      throw new MetadataSerializingException("Unexpected error occurred serializing ObjectType", e);
    }
  }

  @Override
  public void visitString(StringType stringType) {
    writeType(stringType, STRING);
  }


  @Override
  public void visitTime(TimeType timeType) {
    writeType(timeType, TIME);
  }

  @Override
  public void visitTuple(TupleType tupleType) {
    writeType(tupleType, TUPLE);
    try {
      writer.name(OF);
      writer.beginArray();
      final List<MetadataType> types = tupleType.getTypes();
      for (MetadataType type : types) {
        write(type);
      }
      writer.endArray();
    } catch (IOException e) {
      throw new MetadataSerializingException(String.format(UNEXPECTED_ERROR_OCCURRED_SERIALIZING, TUPLE), e);
    }
  }

  @Override
  public void visitFunction(FunctionType functionType) {
    writeType(functionType, FUNCTION);
    final Collection<FunctionParameter> parameters = functionType.getParameters();
    try {
      writer.name(PARAMETERS);
      writer.beginArray();

      for (FunctionParameter parameter : parameters) {
        writer.beginObject();
        writer.name(NAME).value(parameter.getName());
        writer.name(TYPE);
        write(parameter.getType());
        writer.endObject();
      }
      writer.endArray();

      functionType.getReturnType().ifPresent(returnType -> {
        try {
          writer.name(RETURN_TYPE);
          write(returnType);
        } catch (IOException e) {
          throw new MetadataSerializingException("Unexpected error occurred serializing the returnType field for FunctionType",
                                                 e);
        }
      });
    } catch (IOException e) {
      throw new MetadataSerializingException("Unexpected error occurred serializing FunctionType", e);
    }
  }

  @Override
  public void visitUnion(UnionType unionType) {
    writeType(unionType, UNION);
    try {
      writer.name(OF);
      writer.beginArray();
      final List<MetadataType> types = unionType.getTypes();
      for (MetadataType type : types) {
        write(type);
      }
      writer.endArray();
    } catch (IOException e) {
      throw new MetadataSerializingException(String.format(UNEXPECTED_ERROR_OCCURRED_SERIALIZING, UNION), e);
    }
  }

  @Override
  public void visitIntersection(IntersectionType intersectionType) {
    writeType(intersectionType, INTERSECTION);
    try {
      writer.name(OF);
      writer.beginArray();
      final List<MetadataType> types = intersectionType.getTypes();
      for (MetadataType type : types) {
        write(type);
      }
      writer.endArray();
    } catch (IOException e) {
      throw new MetadataSerializingException(String.format(UNEXPECTED_ERROR_OCCURRED_SERIALIZING, INTERSECTION), e);
    }
  }

  private void writeType(MetadataType metadataType, String type) {
    try {
      writer.name(TYPE).value(type);
      writeAnnotations(metadataType.getAnnotations());
    } catch (IOException e) {
      throw new MetadataSerializingException(String.format(UNEXPECTED_ERROR_OCCURRED_SERIALIZING, type), e);
    }
  }

  private void writeAnnotations(Collection<TypeAnnotation> annotations) throws IOException {
    List<TypeAnnotation> publicAnnotations = annotations
        .stream()
        .filter(TypeAnnotation::isPublic)
        .collect(toList());

    if (!publicAnnotations.isEmpty()) {
      writer.name(ANNOTATIONS);
      writer.beginObject();

      for (TypeAnnotation annotation : publicAnnotations) {
        writer.name(getAnnotationJsonName(annotation));
        typeAnnotationSerializer.serialize(writer, annotation);
      }
      writer.endObject();
    }
  }

  private String getAnnotationJsonName(TypeAnnotation annotation) {
    if (typeAnnotationSerializer.getNameClassMapping().containsKey(annotation.getName())) {
      return annotation.getName();
    }
    return annotation.getClass().getName();
  }

  private void createKeyObject(Collection<TypeAnnotation> keyAnnotations, ObjectKeyType key,
                               HashMap<String, String> additionalProperties)
      throws IOException {
    writer.name(KEY);
    writer.beginObject();
    writer.name(NAME);
    String keyString;
    if (key.isName()) {
      keyString = key.getName().toString();
    } else {
      keyString = key.getPattern().toString();
      final RegexPatternAnnotation patternAnnotation = new RegexPatternAnnotation(keyString);
      if (!keyAnnotations.contains(patternAnnotation)) {
        keyAnnotations.add(patternAnnotation);
      }
    }
    writer.value(keyString);
    writeAnnotations(keyAnnotations);
    createAttributes(key);
    for (Map.Entry<String, String> property : additionalProperties.entrySet()) {
      writer.name(property.getKey()).value(property.getValue());
    }

    writer.endObject();
  }

  private void createAttributes(ObjectKeyType key) throws IOException {
    final Collection<AttributeFieldType> attributes = key.getAttributes();
    if (!attributes.isEmpty()) {
      writer.name(ATTRIBUTES);
      writer.beginArray();
      for (AttributeFieldType attribute : attributes) {
        createAttribute(attribute);
      }
      writer.endArray();
    }
  }

  private Map<String, String> getAttributeAdditionalProperties(AttributeFieldType attribute) {
    final HashMap<String, String> result = new HashMap<>();
    if (attribute.isRequired()) {
      result.put(REQUIRED, TRUE);
    }
    return result;
  }

  private void createAttribute(AttributeFieldType attribute) throws IOException {
    writer.beginObject();
    final AttributeKeyType attributeKey = attribute.getKey();
    createAttributeKeyObject(attributeKey, getAttributeAdditionalProperties(attribute));
    writer.name(MODEL);
    write(attribute.getValue());
    writeAnnotations(attribute.getAnnotations());
    writer.endObject();
  }

  private void createAttributeKeyObject(AttributeKeyType key,
                                        Map<String, String> additionalProperties)
      throws IOException {
    writer.name(KEY);
    writer.beginObject();
    writer.name(NAME);
    String keyString;
    Collection<TypeAnnotation> keyAnnotations = new ArrayList<>();
    if (key.isName()) {
      keyString = key.getName().toString();
    } else {
      keyString = key.getPattern().toString();
      final RegexPatternAnnotation patternAnnotation = new RegexPatternAnnotation(keyString);
      if (!keyAnnotations.contains(patternAnnotation)) {
        keyAnnotations.add(patternAnnotation);
      }
    }
    writer.value(keyString);
    writeAnnotations(keyAnnotations);
    for (Map.Entry<String, String> property : additionalProperties.entrySet()) {
      writer.name(property.getKey()).value(property.getValue());
    }
    writer.endObject();
  }

}
