/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.api;


import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.model.MetadataType;

import java.util.Collection;
import java.util.Map;

/**
 * Interface to be implemented to extend the functionality of {@link MetadataType} serialization. The implementation of
 * this class is highly recommended when using custom {@link TypeAnnotation}s in a {@link MetadataType} description to be
 * able to create more human readable serialization output. This implementations will be located using SPI.
 *
 * @since 1.0
 */
public interface TypeAnnotationSerializerExtender {

  /**
   * The purpose of this registry is to provide a mapping between a "friendly name" ({@link TypeAnnotation#getName()}) and
   * the correspondent concrete class ({@link TypeAnnotation#getClass()}), this allows to the serialization to write a more
   * compact and easy to read name, instead of using the full qualifier name of the concrete class.
   * <p />
   * <b>Using this mapping the serialization goes from:</b>
   * <pre>{"org.mule.metadata.api.annotation.TypeIdAnnotation":"java.lang.String"}</pre>
   * <b>To:</b>
   * <pre>{"typeId":"java.lang.String"}</pre>
   * By default the {@link MetadataType} serialization will provide mappings for out of the box {@link TypeAnnotation}
   * implementations as {@link TypeIdAnnotation}, {@link LabelAnnotation}, etc.
   * <p />
   * <b>Considerations:</b>
   * <ol>
   * <li>The name used as Key of the map for a class, should be the same in {@link TypeAnnotation#getName()}</li>
   * <li>If a mapping is not found for a given {@link TypeAnnotation} the full qualifier name will be used</li>
   * <li>If a mapping tries to use a taken name from an already provided {@link TypeAnnotation} implementation, the
   * serialization will fail</li>
   * <li>If the class defined for a {@link TypeAnnotation} implementation is not found in the in the ClassLoader, the
   * deserialization of this object will be ignored.</li>
   *
   * @return a map with the mapping classes by type annotation name.
   */
  default Map<String, Class<? extends TypeAnnotation>> getNameClassMapping() {
    return emptyMap();
  }

  /**
   * Provides an unspecific list of additional serialization features.
   *
   * Because this interface is meant to remain agnostic of the actual serialization implementation, these features
   * are represented as simply objects in order to avoid exposing internal classes.
   *
   * Also, because these features might be implementation specific, the consumer of this class is free to skip the features
   * it cannot handle. Thus, the features returned here are to be considered more like a suggestion to the serializer than
   * an actual directive.
   *
   * @return a list with additional serialization features
   */
  default Collection<Object> getAdditionalFeatures() {
    return emptyList();
  }
}
