/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.metadata.persistence.MetadataTypeConstants.REF_FLAG;
import static org.mule.metadata.persistence.MetadataTypeConstants.TYPE;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.ObjectType;

import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.util.Optional;

/**
 * Implementation of {@link ObjectTypeReferenceHandler} which handles references
 * by looking for the {@link MetadataTypeConstants#REF_FLAG} prefix.
 *
 * @since 1.0
 */
public class DefaultObjectTypeReferenceHandler implements ObjectTypeReferenceHandler {

  private final SerializationContext serializationContext;

  /**
   * Creates a new instance
   * @param serializationContext a {@link SerializationContext} to keep track of the references
   */
  public DefaultObjectTypeReferenceHandler(SerializationContext serializationContext) {
    this.serializationContext = serializationContext;
  }

  /**
   * If {@code typeReference} has the {@link MetadataTypeConstants#REF_FLAG} then
   * it returns the type registered on the {@link #serializationContext} under such
   * key.
   *
   * @param typeReference a reference
   * @return an {@link Optional} {@link TypeBuilder}
   */
  @Override
  public Optional<TypeBuilder> readReference(String typeReference) {
    if (typeReference.startsWith(REF_FLAG)) {
      return serializationContext.getRegisteredObjectType(typeReference.replace(REF_FLAG, ""))
          .map(type -> (TypeBuilder) () -> type);
    }

    return empty();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<String> writeReference(ObjectType type, JsonWriter writer) {
    String serializationId = serializationContext.registerObjectType(type);
    try {
      writer.name(TYPE).value(REF_FLAG + serializationId);
    } catch (IOException e) {
      throw new MetadataSerializingException("Unexpected error occurred writing type reference", e);
    }

    return of(serializationId);
  }

}
