/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.deserializer;

import static org.mule.metadata.persistence.MetadataTypeConstants.ATTRIBUTES;
import static org.mule.metadata.persistence.MetadataTypeConstants.FIELDS;
import static org.mule.metadata.persistence.MetadataTypeConstants.KEY;
import static org.mule.metadata.persistence.MetadataTypeConstants.MODEL;
import static org.mule.metadata.persistence.MetadataTypeConstants.NAME;
import static org.mule.metadata.persistence.MetadataTypeConstants.OPEN;
import static org.mule.metadata.persistence.MetadataTypeConstants.ORDERED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REPEATED;
import static org.mule.metadata.persistence.MetadataTypeConstants.REQUIRED;
import static org.mule.metadata.persistence.deserializer.SerializerUtils.VOLATILE_FORMAT;
import static org.mule.metadata.persistence.deserializer.SerializerUtils.getAnnotations;

import org.mule.metadata.api.annotation.RegexPatternAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.builder.AttributeFieldTypeBuilder;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectKeyBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.persistence.JsonMetadataTypeLoader;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/**
 * {@link TypeDeserializer} implementation that deserializes an {@link ObjectType} and returns an {@link ObjectTypeBuilder}
 *
 * @since 1.0
 */
final class ObjectTypeDeserializer extends AbstractTypeDeserializer {

  private static final Pattern QNAME_PATTERN = Pattern.compile("(?:\\{([\\d\\w\\D\\W]*)\\})?([\\d\\w\\D\\W]*)");

  ObjectTypeDeserializer() {
    super(BaseTypeBuilder::objectType);
  }

  @Override
  public TypeBuilder buildType(JsonObject typeObject, BaseTypeBuilder baseBuilder, JsonMetadataTypeLoader typeLoader) {

    final ObjectTypeBuilder objectTypeBuilder = (ObjectTypeBuilder) supplier.get(baseBuilder);
    if (typeObject.has(ORDERED)) {
      objectTypeBuilder.ordered(typeObject.get(ORDERED).getAsBoolean());
    }

    if (typeObject.has(OPEN)) {
      objectTypeBuilder.openWith(typeLoader.buildType(typeObject.get(OPEN), objectTypeBuilder.openWith()));
    }

    final JsonElement fields = typeObject.get(FIELDS);
    List<TypeAnnotation> annotations = getAnnotations(typeObject);
    annotations.forEach(objectTypeBuilder::with);

    for (JsonElement jsonElement : fields.getAsJsonArray()) {
      final JsonObject field = jsonElement.getAsJsonObject();
      final ObjectFieldTypeBuilder fieldBuilder;
      final JsonObject key = field.get(KEY).getAsJsonObject();

      final String keyName = key.get(NAME).getAsString();
      if (isPattern(key)) {
        fieldBuilder = objectTypeBuilder.addField().key(Pattern.compile(keyName));
      } else {
        fieldBuilder = objectTypeBuilder.addField();
        final ObjectKeyBuilder keyBuilder = fieldBuilder.key(parseKey(keyName));
        if (key.has(ATTRIBUTES)) {
          parseAttributes(key, keyBuilder, typeLoader, VOLATILE_FORMAT);
        }
      }
      if (key.has(REQUIRED)) {
        fieldBuilder.required(key.get(REQUIRED).getAsBoolean());
      }
      if (key.has(REPEATED)) {
        fieldBuilder.repeated(key.get(REPEATED).getAsBoolean());
      }
      getAnnotations(key).forEach(fieldBuilder::withKeyAnnotation);
      getAnnotations(field).forEach(fieldBuilder::with);
      final JsonElement valueElement = field.get(MODEL);
      if (valueElement.isJsonObject()) {
        final JsonObject valueObject = valueElement.getAsJsonObject();
        fieldBuilder.value(typeLoader.buildType(valueObject, new BaseTypeBuilder(VOLATILE_FORMAT)));
      } else {
        fieldBuilder.value(typeLoader.buildType(valueElement, fieldBuilder.value()));
      }
    }
    return objectTypeBuilder;
  }

  private void parseAttributes(JsonObject value, ObjectKeyBuilder objectKeyBuilder, JsonMetadataTypeLoader typeLoader,
                               MetadataFormat format) {
    for (JsonElement attributeJsonElement : value.get(ATTRIBUTES).getAsJsonArray()) {
      final AttributeFieldTypeBuilder attributeFieldTypeBuilder = objectKeyBuilder.addAttribute();

      final JsonObject attribute = attributeJsonElement.getAsJsonObject();

      final JsonObject attributeKey = attribute.get(KEY).getAsJsonObject();

      final String attributeKeyName = attributeKey.get(NAME).getAsString();
      if (isPattern(attributeKey)) {
        attributeFieldTypeBuilder.pattern(Pattern.compile(attributeKeyName));
      } else {
        attributeFieldTypeBuilder.name(parseKey(attributeKeyName));
      }
      if (attributeKey.has(REQUIRED)) {
        attributeFieldTypeBuilder.required(attributeKey.get(REQUIRED).getAsBoolean());
      }
      final JsonObject attributeModel = attribute.get(MODEL).getAsJsonObject();
      attributeFieldTypeBuilder.value(typeLoader.buildType(attributeModel, new BaseTypeBuilder(format)));
    }
  }

  private QName parseKey(String key) {
    final Matcher matcher = QNAME_PATTERN.matcher(key);
    if (matcher.find()) {
      if (matcher.groupCount() == 1) {
        return new QName(key);
      }
      if (matcher.groupCount() == 2) {
        return new QName(matcher.group(1), matcher.group(2));
      }
    }
    return new QName(key);
  }

  private boolean isPattern(JsonElement value) {
    if (value.isJsonObject()) {
      for (TypeAnnotation typeAnnotation : getAnnotations(value.getAsJsonObject())) {
        if (typeAnnotation.getName().equals(RegexPatternAnnotation.NAME)) {
          return true;
        }
      }
    }

    return false;
  }
}
