/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.type.adapter;

import static java.util.Optional.ofNullable;

import static com.google.gson.JsonParser.parseReader;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Optional;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.reflect.TypeToken;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

/**
 * {@link TypeAdapter<Optional>} which overrides the default behaviour of {@link Gson} to be able to write more clean and less
 * verbose JSON. This {@link TypeAdapter} unwraps the Optional value and serializes it, when deserializing the value gets wrapped
 * to comply with the Java types.
 * <p>
 * <b>Using default GSON TypeAdapter:</b>
 *
 * <pre>
 * {"optionalValue":{"value":"this is the value"}}
 * </pre>
 *
 * <b>Using OptionalTypeAdapter:</b>
 *
 * <pre>
 * {"optionalValue":"this is the value"}
 * </pre>
 *
 * @param <T> Type of the Optional value
 * @since 1.0
 */
final class OptionalTypeAdapter<T> extends TypeAdapter<Optional<T>> {

  private final Gson gson;
  private final Type innerType;

  OptionalTypeAdapter(Gson gson, Type innerType) {
    this.gson = gson;
    this.innerType = innerType;
  }

  @Override
  public void write(JsonWriter out, Optional<T> value) throws IOException {
    if (value != null && value.isPresent()) {
      @SuppressWarnings("unchecked")
      TypeAdapter<T> adapter = (TypeAdapter<T>) gson.getAdapter(TypeToken.get(innerType));
      adapter.write(out, value.get());
    } else {
      out.nullValue();
    }
  }

  @Override
  public Optional<T> read(JsonReader in) throws IOException {
    @SuppressWarnings("unchecked")
    TypeAdapter<T> adapter = (TypeAdapter<T>) gson.getAdapter(TypeToken.get(innerType));
    final T t = adapter.fromJson(parseReader(in).toString());

    return ofNullable(t);
  }
}
